#include "evolution_policy.h"

#include <drive/backend/common/localization.h>
#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/data/chargable.h>
#include <drive/backend/data/event_tag.h>
#include <drive/backend/data/evolution_policy.h>
#include <drive/backend/data/feedback.h>
#include <drive/backend/data/taxi.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/transaction/assert.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/device_snapshot/snapshots/image.h>
#include <drive/backend/offers/action.h>
#include <drive/backend/roles/manager.h>
#include <drive/backend/sessions/manager/billing.h>

TUserAction::TFactory::TRegistrator<TCovidVoluneteersEvolutionPolicyAction> TCovidVoluneteersEvolutionPolicyAction::Registrator(TCovidVoluneteersEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TRequiredPhotosEvolutionPolicyAction> TRequiredPhotosEvolutionPolicyAction::Registrator(TRequiredPhotosEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TSelfieVerificationEvolutionPolicyAction> TSelfieVerificationEvolutionPolicyAction::Registrator(TSelfieVerificationEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TRideCompletionEvolutionPolicyAction> TRideCompletionEvolutionPolicyAction::Registrator(TRideCompletionEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TTaxiRouteHistoryEvolutionPolicyAction> TTaxiRouteHistoryEvolutionPolicyAction::Registrator(TTaxiRouteHistoryEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TTaxisharingEvolutionPolicyAction> TTaxisharingEvolutionPolicyAction::Registrator(TTaxisharingEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TTelematicsCommandEvolutionPolicyAction> TTelematicsCommandEvolutionPolicyAction::Registrator(TTelematicsCommandEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TUserProximityEvolutionPolicyAction> TUserProximityEvolutionPolicyAction::Registrator(TUserProximityEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TCloseChatEvolutionPolicyAction> TCloseChatEvolutionPolicyAction::Registrator(TCloseChatEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TMoveChatEvolutionPolicyAction> TMoveChatEvolutionPolicyAction::Registrator(TMoveChatEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TScooterFixPointEvolutionPolicyAction> TScooterFixPointEvolutionPolicyAction::Registrator(TScooterFixPointEvolutionPolicyAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TBlockSessionEvolutionPolicyAction> TBlockSessionEvolutionPolicyAction::Registrator(TBlockSessionEvolutionPolicyAction::GetTypeName());

ITag::TAggregateEvolutionPolicy TCovidVoluneteersEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode /*eMode*/, NDrive::TEntitySession& /*session*/) const {
    if (newTag->GetName() != TChargableTag::Acceptance && newTag->GetName() != TChargableTag::Riding) {
        return ITag::TAggregateEvolutionPolicy();
    }
    if (dbTag->GetName() != TChargableTag::Reservation && dbTag->GetName() != TChargableTag::Parking) {
        return ITag::TAggregateEvolutionPolicy();
    }
    ITag::TAggregateEvolutionPolicy result;

    auto oldTagImpl = dbTag.GetTagAs<TChargableTag>();
    if (!oldTagImpl) {
        return ITag::TAggregateEvolutionPolicy();
    }
    auto offer = oldTagImpl->GetOffer();
    if (!offer) {
        offer = server->GetDriveAPI()->GetCurrentCarOffer(dbTag.GetObjectId());
    }
    if (!offer) {
        return ITag::TAggregateEvolutionPolicy();
    }

    bool checkPermit = false;
    auto action = server->GetDriveAPI()->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
    auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
    if (offerBuilder) {
        const auto& tags = offerBuilder->GetGrouppingTags();
        checkPermit = tags.contains(TariffActionTag);
    }
    if (checkPermit) {
        result.Policies.emplace_back(new TPassCheckEvolutionPolicy(PermitTagName, PermitLandingId, WarningChatId, IgnoreTagName));
    }

    return result;
}

bool TCovidVoluneteersEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TJsonProcessor::Read(jsonValue, "tariff_action_tag", TariffActionTag)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "permit_tag_name", PermitTagName)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "permit_landing_id", PermitLandingId)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "warning_chat_id", WarningChatId)) {
        return false;
    }
    TJsonProcessor::Read(jsonValue, "ignore_tag_name", IgnoreTagName);
    return true;
}

NJson::TJsonValue TCovidVoluneteersEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue jsonValue;
    TJsonProcessor::Write(jsonValue, "tariff_action_tag", TariffActionTag);
    TJsonProcessor::Write(jsonValue, "permit_tag_name", PermitTagName);
    TJsonProcessor::Write(jsonValue, "permit_landing_id", PermitLandingId);
    TJsonProcessor::Write(jsonValue, "ignore_tag_name", IgnoreTagName);
    TJsonProcessor::Write(jsonValue, "warning_chat_id", WarningChatId);
    return jsonValue;
}

NDrive::TScheme TCovidVoluneteersEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("tariff_action_tag", "Тег тарифа для запроса пропуска");
    result.Add<TFSString>("permit_tag_name", "Тег означающий наличие пропуска");
    result.Add<TFSString>("permit_landing_id", "Лендинг недопуска");
    result.Add<TFSString>("ignore_tag_name", "Тег для игнорирования проверок");
    result.Add<TFSString>("warning_chat_id", "Чат с проверкой пропуска");
    return result;
}

TString TCovidVoluneteersEvolutionPolicyAction::GetTypeName() {
    return "covid_volunteers_pass_check";
}

TString TCovidVoluneteersEvolutionPolicyAction::GetType() const {
    return GetTypeName();
}

ITag::TAggregateEvolutionPolicy TRequiredPhotosEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, EEvolutionMode eMode, NDrive::TEntitySession& tx) const {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    Y_UNUSED(eMode);
    auto evlog = NDrive::GetThreadEventLogger();
    if (!dbTag.Is<TChargableTag>()) {
        return {};
    }
    if (Yensured(newTag)->GetName() != TChargableTag::Reservation) {
        return {};
    }
    if (eMode != EEvolutionMode::Default) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "SkipRequiredPhotosEvolutionPolicy")
                ("evolution_mode", ToString(eMode))
            );
        }
        return {};
    }
    if (tx.GetOriginatorId()) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "SkipRequiredPhotosEvolutionPolicy")
                ("originator_id", tx.GetOriginatorId())
            );
        }
        return {};
    }
    if (!PhotoTagNames.empty()) {
        auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetTagSession(dbTag.GetTagId(), tx);
        R_ENSURE(optionalSession, {}, "cannot GetTagSession " << dbTag.GetTagId(), tx);
        auto session = *optionalSession;
        R_ENSURE(session, HTTP_INTERNAL_SERVER_ERROR, "cannot find session from tag " << dbTag.GetTagId(), tx);
        auto sessionId = session->GetSessionId();
        auto optionalTags = server->GetDriveDatabase().GetTagsManager().GetTraceTags().RestoreTagsRobust({ sessionId }, PhotoTagNames, tx);
        R_ENSURE(optionalTags, {}, "cannot RestoreTags " << sessionId, tx);
        ui32 currentPhotoCount = 0;
        for (auto&& tag : *optionalTags) {
            auto snapshot = tag->GetObjectSnapshotAs<TImagesSnapshot>();
            if (snapshot) {
                currentPhotoCount += snapshot->GetImages().size();
            }
        }
        if (currentPhotoCount < PhotoCount) {
            tx.SetErrorDetails(NJson::TMapBuilder
                ("required_photo_count", PhotoCount - currentPhotoCount)
                ("required_tags", NJson::ToJson(PhotoTagNames))
            );
            return ITag::TAggregateEvolutionPolicy::CreateError(TStringBuilder() << "only " << currentPhotoCount << " of " << PhotoCount << " photos are present")
                .SetError(NDrive::MakeError("photos_required_on_evolution"))
                .SetErrorCode(HTTP_BAD_REQUEST)
            ;
        }
    }
    return {};
}

bool TRequiredPhotosEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["photo_count"], PhotoCount) &&
        NJson::ParseField(value["photo_tag_names"], PhotoTagNames);
}

NJson::TJsonValue TRequiredPhotosEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "photo_count", PhotoCount);
    NJson::InsertField(result, "photo_tag_names", PhotoTagNames);
    return result;
}

NDrive::TScheme TRequiredPhotosEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSNumeric>("photo_count").SetDefault(PhotoCount);
    result.Add<TFSVariants>("photo_tag_names").SetMultiSelect(true).SetVariants(
        server->GetDriveDatabase().GetTagsManager().GetTagsMeta().GetRegisteredTagNames({ TFeedbackTraceTag::Type() })
    );
    return result;
}

ITag::TAggregateEvolutionPolicy TSelfieVerificationEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode /*eMode*/, NDrive::TEntitySession& session) const {
    if (newTag->GetName() != TChargableTag::Riding && newTag->GetName() != TChargableTag::Reservation || dbTag->GetName() != TChargableTag::Acceptance) {
        return ITag::TAggregateEvolutionPolicy();
    }

    ITag::TAggregateEvolutionPolicy result;

    auto optionalTaggedUser = server->GetDriveAPI()->GetTagsManager().GetUserTags().RestoreObject(permissions.GetUserId(), session);
    if (!optionalTaggedUser) {
        return {};
    }
    if (VerificationTagName) {
        auto optionalTag = optionalTaggedUser->GetTag(VerificationTagName);
        if (!optionalTag) {
            return result;
        }
    }
    if (EventTagName && newTag->GetName() == TChargableTag::Reservation) {
        auto optionalEventTag = optionalTaggedUser->GetTag(EventTagName);
        auto eventTag = optionalEventTag ? optionalEventTag->MutableTagAs<IEventTag>() : nullptr;
        if (eventTag) {
            eventTag->AddEvent(newTag->GetName(), Now());
            if (!server->GetDriveAPI()->GetTagsManager().GetUserTags().UpdateTagData(*optionalEventTag, permissions.GetUserId(), session)) {
                return ITag::TAggregateEvolutionPolicy::CreateError("Could not update event tag " + session.GetStringReport());
            }
        }
    }

    if (newTag->GetName() == TChargableTag::Riding && session.GetRequestContext().GetUserChoice() != "accept") {
        result.Policies.emplace_back(new TAskForSelfiePolicy(VerificationLandingId));
    }

    return result;
}

bool TSelfieVerificationEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    return
        NJson::ParseField(jsonValue, "verification_tag_name", VerificationTagName) &&
        NJson::ParseField(jsonValue, "recheck_documents_tag_name", RecheckDocumentsTagName) &&
        NJson::ParseField(jsonValue, "verification_landing_id", VerificationLandingId) &&
        NJson::ParseField(jsonValue, "event_tag_name", EventTagName) &&
        NJson::ParseField(jsonValue, "use_antifraud", UseAntifraud, false) &&
        NJson::ParseField(jsonValue, "wait_for_antifraud", WaitForAntifraud, false) &&
        NJson::ParseField(jsonValue, "tags_to_add", TagsToAddOnSubmit);
}

NJson::TJsonValue TSelfieVerificationEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue jsonValue;
    NJson::InsertField(jsonValue, "verification_tag_name", VerificationTagName);
    NJson::InsertField(jsonValue, "verification_landing_id", VerificationLandingId);
    NJson::InsertField(jsonValue, "recheck_documents_tag_name", RecheckDocumentsTagName);
    NJson::InsertField(jsonValue, "event_tag_name", EventTagName);
    NJson::InsertField(jsonValue, "tags_to_add", TagsToAddOnSubmit);
    NJson::InsertField(jsonValue, "use_antifraud", UseAntifraud);
    NJson::InsertField(jsonValue, "wait_for_antifraud", WaitForAntifraud);
    return jsonValue;
}

NDrive::TScheme TSelfieVerificationEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("verification_tag_name", "Тег для запроса верификации по селфи");
    result.Add<TFSString>("verification_landing_id", "Лендинг для верификации");
    result.Add<TFSString>("recheck_documents_tag_name", "Тег для отправки документов на перепроверку");
    result.Add<TFSString>("event_tag_name", "Тег для записи событий");
    result.Add<TFSArray>("tags_to_add", "Навесить теги при сабмите селфи").SetElement<TFSText>();
    result.Add<TFSBoolean>("use_antifraud", "Использовать автоматическое распознование селфи");
    result.Add<TFSBoolean>("wait_for_antifraud", "Ждать результатов распознования селфи");
    return result;
}

TString TSelfieVerificationEvolutionPolicyAction::GetTypeName() {
    return "selfie_verification";
}

TString TSelfieVerificationEvolutionPolicyAction::GetType() const {
    return GetTypeName();
}

ITag::TAggregateEvolutionPolicy TRideCompletionEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    Y_UNUSED(session);
    if (newTag->GetName() != TChargableTag::Reservation) {
        return ITag::TAggregateEvolutionPolicy();
    }

    auto chTag = dbTag.GetTagAs<TChargableTag>();
    if (!chTag) {
        return ITag::TAggregateEvolutionPolicy();
    }

    ITag::TAggregateEvolutionPolicy result;
    auto delegationPolicyPtr = MakeHolder<TCarDelegationPolicy>(permissions.GetUserId());
    delegationPolicyPtr->SetFreeDelegationConfirmationLanding(FreeDelegationConfirmationLanding);
    delegationPolicyPtr->SetP2PDelegationDuration(P2PDelegationDuration);

    NJson::TJsonValue payloadPatch;
    NJson::TJsonValue delegationParams;
    TMaybe<ECarDelegationType> delegationType;
    bool hasPayloadPatch = requestData.Has("payload_patch") && requestData["payload_patch"].IsString() && NJson::ReadJsonTree(requestData["payload_patch"].GetString(), &payloadPatch);

    if (requestData.Has("delegation")) {
        delegationParams = requestData["delegation"];
    } else if (payloadPatch.Has("delegation")) {
        delegationParams = payloadPatch["delegation"];
    }
    if (delegationParams.IsDefined()) {
        ECarDelegationType dType;
        if (TryFromString(delegationParams["type"].GetString(), dType)) {
            delegationType = dType;
        }
    }

    // Delegation types, which can take place in the allowed drop area
    {
        if (delegationType.Defined() && *delegationType != ECarDelegationType::Free) {
            result.NewTag = chTag->Clone(server->GetDriveAPI()->GetTagsManager());
            if (!result.NewTag) {
                return ITag::TAggregateEvolutionPolicy::CreateError("Could not copy tag");
            }
            result.NewTag->SetName(TChargableTag::Parking);

            if (!delegationPolicyPtr->DeserializeFromJson(delegationParams)) {
                return ITag::TAggregateEvolutionPolicy::CreateError("Could not construct p2p delegation");
            }

            ITag::TAggregateEvolutionPolicy policyWithError;
            if (!delegationPolicyPtr->CheckDelegationPossibleFromTag(dbTag, policyWithError)) {
                return policyWithError;
            }

            result.Policies.push_back(std::move(delegationPolicyPtr));

            return result;
        }
    }

    // Delegation types, which shall take place outside allowed drop area (so-called "suspended rent")
    TRTDeviceSnapshot snapshot = server->GetSnapshotsManager().GetSnapshot(dbTag.GetObjectId());
    TCarLocationContext context(dbTag.GetObjectId(), snapshot);

    const bool force = eMode != EEvolutionMode::Default || snapshot.GetImei().empty();
    const TCarLocationFeatures clf = context.GetCarAreaFeatures(force, server, TInstant::Zero());
    result.Policies.push_back(new TRideFinishPolicy(clf, dbTag.GetTagId()));
    if (clf.GetAllowDropDef(EDropAbility::NotAllow) == EDropAbility::DenyIncorrectData && dbTag->GetName() != TChargableTag::Reservation && CheckIncorrectSignal) {
        NDrive::TEventLog::Log("EvolutionPolicyIncorrectCarSignal", NJson::TMapBuilder
            ("object_id", dbTag.GetObjectId())
            ("location", NJson::ToJson(clf.OptionalLocation()))
        );
        return ITag::TAggregateEvolutionPolicy::CreateError("No GPS signal")
            .SetErrorCode(HTTP_NOT_ACCEPTABLE)
            .SetResult(EDriveSessionResult::IncorrectCarSignal)
        ;
    }
    if (clf.GetAllowDropDef(EDropAbility::NotAllow) == EDropAbility::NotAllow && FreeDelegationEnabled) {
        result.NewTag = chTag->Clone(server->GetDriveAPI()->GetTagsManager());
        if (!result.NewTag) {
            return ITag::TAggregateEvolutionPolicy::CreateError("Could not copy tag");
        }
        result.NewTag->SetName(TChargableTag::Parking);

        if (hasPayloadPatch && delegationType.Defined() && !delegationPolicyPtr->DeserializeFromJson(delegationParams)) {
            return ITag::TAggregateEvolutionPolicy::CreateError("Could not deserialize delegation params from json");
        }
        result.Policies.clear();
        result.Policies.push_back(delegationPolicyPtr.Release());
        return result;
    }

    return result;
}

bool TRideCompletionEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    if (!TJsonProcessor::Read(jsonValue, "free_delegation_enabled", FreeDelegationEnabled)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "free_delegation_confirmation_landing", FreeDelegationConfirmationLanding)) {
        return false;
    }
    if (!TJsonProcessor::Read(jsonValue, "p2p_delegation_duration", P2PDelegationDuration)) {
        return false;
    }
    return NJson::ParseField(jsonValue["check_incorrect_signal"], CheckIncorrectSignal);
}

NJson::TJsonValue TRideCompletionEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue jsonValue;
    NJson::InsertField(jsonValue, "check_incorrect_signal", CheckIncorrectSignal);
    TJsonProcessor::Write(jsonValue, "free_delegation_enabled", FreeDelegationEnabled);
    TJsonProcessor::Write(jsonValue, "free_delegation_confirmation_landing", FreeDelegationConfirmationLanding);
    TJsonProcessor::Write(jsonValue, "p2p_delegation_duration", P2PDelegationDuration);
    return jsonValue;
}

NDrive::TScheme TRideCompletionEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSBoolean>("check_incorrect_signal", "Enable 'No GPS signal' check");
    result.Add<TFSBoolean>("free_delegation_enabled", "Включить подвешенную аренду");
    result.Add<TFSString>("free_delegation_confirmation_landing", "Лендинг подтверждения подвешенной аренды");
    result.Add<TFSDuration>("p2p_delegation_duration", "Время на передачу руля");
    return result;
}

TString TRideCompletionEvolutionPolicyAction::GetTypeName() {
    return "ride_completion";
}

TString TRideCompletionEvolutionPolicyAction::GetType() const {
    return GetTypeName();
}

ITag::TAggregateEvolutionPolicy TTaxiRouteHistoryEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    Y_UNUSED(server);
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    Y_UNUSED(eMode);
    Y_UNUSED(session);
    if (!dbTag.Is<TChargableTag>()) {
        return {};
    }
    if (Yensured(newTag)->GetName() != TChargableTag::Reservation) {
        return {};
    }
    ITag::TAggregateEvolutionPolicy result;
    result.Policies.push_back(MakeAtomicShared<TTaxiRouteHistoryPolicy>());
    return result;
}

ITag::TAggregateEvolutionPolicy TTaxisharingEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    Y_UNUSED(eMode);
    if (!dbTag.Is<TChargableTag>()) {
        return {};
    }
    if (Yensured(newTag)->GetName() != TChargableTag::Reservation) {
        return {};
    }
    if (OfferTagsFilter) {
        auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetTagSession(dbTag.GetTagId(), session);
        R_ENSURE(optionalSession, {}, "cannot GetTagSession " << dbTag.GetTagId(), session);
        auto billingSession = std::dynamic_pointer_cast<const TBillingSession>(*optionalSession);
        auto currentOffer = billingSession ? billingSession->GetCurrentOffer() : nullptr;
        auto action = currentOffer ? server->GetDriveDatabase().GetUserPermissionManager().GetAction(currentOffer->GetBehaviourConstructorId()) : Nothing();
        auto builder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
        if (!builder || !OfferTagsFilter.IsMatching(builder->GetGrouppingTags())) {
            return {};
        }
    }
    ITag::TAggregateEvolutionPolicy result;
    result.Policies.push_back(MakeAtomicShared<TTaxisharingPolicy>(FuelLevelThreshold));
    return result;
}

bool TTaxisharingEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["fuel_level_threshold"], FuelLevelThreshold) &&
        NJson::ParseField(value["offer_tags_filter"], OfferTagsFilter);
}

NJson::TJsonValue TTaxisharingEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "fuel_level_threshold", FuelLevelThreshold);
    NJson::InsertField(result, "offer_tags_filter", OfferTagsFilter);
    return result;
}

NDrive::TScheme TTaxisharingEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSNumeric>("fuel_level_threshold").SetDefault(FuelLevelThreshold);
    result.Add<TFSString>("offer_tags_filter", "Фильтр тегов офферов");
    return result;
}

ITag::TAggregateEvolutionPolicy TTelematicsCommandEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    Y_UNUSED(session);
    if (eMode != EEvolutionMode::Default) {
        return {};
    }
    if (dbTag->GetName() != From) {
        return {};
    }
    if (Yensured(newTag)->GetName() != To) {
        return {};
    }
    auto api = Yensured(server)->GetDriveAPI();
    auto offer = Yensured(api)->GetCurrentCarOffer(dbTag.GetObjectId());
    if (!offer) {
        return {};
    }
    auto offerBuilderAction = api->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
    auto offerBuilder = offerBuilderAction ? offerBuilderAction->GetAs<IOfferBuilderAction>() : nullptr;
    if (!offerBuilder) {
        return {};
    }
    for (auto&& offerTag : OfferTags) {
        if (!offerBuilder->GetGrouppingTags().contains(offerTag)) {
            return {};
        }
    }
    ITag::TAggregateEvolutionPolicy result;
    result.Command = MakeAtomicShared<NDrive::NVega::TCommand>(Command);
    return result;
}

bool TTelematicsCommandEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["command"], NJson::Stringify(Command.Code), true) &&
        NJson::ParseField(value["from"], From, true) &&
        NJson::ParseField(value["to"], To, true) &&
        TJsonProcessor::ReadContainer(value, "offer_tags", OfferTags);
}

NJson::TJsonValue TTelematicsCommandEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "command", NJson::Stringify(Command.Code));
    NJson::InsertField(result, "from", From);
    NJson::InsertField(result, "to", To);
    TJsonProcessor::WriteContainerString(result, "offer_tags", OfferTags);
    return result;
}

NDrive::TScheme TTelematicsCommandEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSVariants>("command", "Telematics command").InitVariants<NDrive::NVega::ECommandCode>();
    result.Add<TFSString>("from", "Source tag name").SetRequired(true);
    result.Add<TFSString>("to", "Target tag name").SetRequired(true);
    result.Add<TFSString>("offer_tags", "Required offer tags");
    return result;
}

ITag::TAggregateEvolutionPolicy TUserProximityEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode /*eMode*/, NDrive::TEntitySession& /*session*/) const {
    if (newTag && (newTag->GetName() == TChargableTag::Riding || newTag->GetName() == TChargableTag::Acceptance)) {
        auto userLocation = permissions.GetUserFeatures().GetUserLocationPtr();
        if (!userLocation) {
            if (!IgnoreUndefinedLocation) {
                return ITag::TAggregateEvolutionPolicy::CreateError("user location is undefined", "undefined_user_location", HTTP_NOT_ACCEPTABLE);
            } else {
                return {};
            }
        }
        auto snapshot = server->GetSnapshotsManager().GetSnapshot(dbTag.GetObjectId());
        auto location = snapshot.GetLocation();
        auto distance = location ? location->GetCoord().GetLengthTo(*userLocation) : 0;
        if (distance > MaximalDistance) {
            return ITag::TAggregateEvolutionPolicy::CreateError(
                TStringBuilder() << "user location is too distant: " << distance,
                "error.distant_user_location.message",
                HTTP_BAD_REQUEST
            ).SetUIErrorTitle(
                "error.distant_user_location.title"
            ).SetError(
                NDrive::MakeError("distant_user_location")
            );
        }
    }
    return {};
}

bool TUserProximityEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["ignore_undefined_location"], IgnoreUndefinedLocation) &&
        NJson::ParseField(value["maximal_distance"], MaximalDistance);
}

NJson::TJsonValue TUserProximityEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "ignore_undefined_location", IgnoreUndefinedLocation);
    NJson::InsertField(result, "maximal_distance", MaximalDistance);
    return result;
}

NDrive::TScheme TUserProximityEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSBoolean>("ignore_undefined_location").SetDefault(MaximalDistance);
    result.Add<TFSNumeric>("maximal_distance").SetDefault(MaximalDistance);
    return result;
}

ITag::TAggregateEvolutionPolicy TCloseChatEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    Y_UNUSED(eMode);
    Y_UNUSED(session);
    if (newTag->GetName() != TChargableTag::Reservation || ChatTagNames.empty()) {
        return ITag::TAggregateEvolutionPolicy();
    }
    ITag::TAggregateEvolutionPolicy result;
    TVector<NDrive::NChat::TMessage> messages;
    if (MessageText) {
        NDrive::NChat::TMessage message(MessageText, 0, NDrive::NChat::TMessage::EMessageType::Plaintext);
        messages.emplace_back(std::move(message));
    }
    if (OfferTagsFilter.IsEmpty()) {
        result.Policies.emplace_back(MakeAtomicShared<TCloseChatEvolutionPolicy>(ChatTagNames, messages));
    } else if (auto chargableTag = dbTag.GetTagAs<TChargableTag>()) {
        auto offer = chargableTag->GetOffer();
        if (!offer) {
            offer = server->GetDriveAPI()->GetCurrentCarOffer(dbTag.GetObjectId());
        }
        if (offer) {
            auto action = server->GetDriveAPI()->GetRolesManager()->GetAction(offer->GetBehaviourConstructorId());
            auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
            if (offerBuilder && OfferTagsFilter.IsMatching(offerBuilder->GetGrouppingTags())) {
                result.Policies.emplace_back(MakeAtomicShared<TCloseChatEvolutionPolicy>(ChatTagNames, messages));
            }
        }
    }
    return result;
}

bool TCloseChatEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["chat_tag_names"], ChatTagNames) &&
        NJson::ParseField(value["offer_tags_filter"], OfferTagsFilter) &&
        NJson::ParseField(value["message_text"], MessageText);
}

NJson::TJsonValue TCloseChatEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "chat_tag_names", ChatTagNames);
    NJson::InsertField(result, "offer_tags_filter", OfferTagsFilter);
    NJson::InsertField(result, "message_text", MessageText);
    return result;
}

NDrive::TScheme TCloseChatEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSArray>("chat_tag_names", "Чатовые теги").SetElement<TFSString>();
    result.Add<TFSString>("offer_tags_filter", "Фильтр тегов офферов");
    result.Add<TFSString>("message_text", "Сообщение");
    return result;
}

ITag::TAggregateEvolutionPolicy TMoveChatEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, EEvolutionMode eMode, NDrive::TEntitySession& tx) const {
    Y_UNUSED(permissions);
    Y_UNUSED(requestData);
    Y_UNUSED(eMode);
    Y_UNUSED(tx);
    if (!dbTag.Is<TChargableTag>()) {
        return {};
    }
    if (Yensured(newTag)->GetName() != TargetTagName) {
        return {};
    }
    auto chatRobot = server->GetChatRobot(ChatName);
    if (!chatRobot) {
        return ITag::TAggregateEvolutionPolicy::CreateError("cannot find ChatRobot " + ChatName);
    }
    auto chatTopic = TString{};
    if (!chatRobot->MoveToStep(dbTag->GetPerformer(), chatTopic, NodeName, nullptr, /*sendMessages=*/true)) {
        return ITag::TAggregateEvolutionPolicy::CreateError("cannot MoveToStep");
    }
    return {};
}

bool TMoveChatEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["chat_name"], ChatName) &&
        NJson::ParseField(value["node_name"], NodeName) &&
        NJson::ParseField(value["target_tag_name"], TargetTagName);
}

NJson::TJsonValue TMoveChatEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "chat_name", ChatName);
    NJson::InsertField(result, "node_name", NodeName);
    NJson::InsertField(result, "target_tag_name", TargetTagName);
    return result;
}

NDrive::TScheme TMoveChatEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("chat_name");
    result.Add<TFSString>("node_name");
    result.Add<TFSVariants>("target_tag_name").SetVariants({
        TChargableTag::Reservation,
        TChargableTag::Acceptance,
        TChargableTag::Riding,
        TChargableTag::Parking,
    });
    return result;
}

ITag::TAggregateEvolutionPolicy TScooterFixPointEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode eMode, NDrive::TEntitySession& /*session*/) const {
    if (!dbTag.Is<TChargableTag>()) {
        return {};
    }
    if (Yensured(newTag)->GetName() != TChargableTag::Reservation) {
        return {};
    }

    TRTDeviceSnapshot snapshot = server->GetSnapshotsManager().GetSnapshot(dbTag.GetObjectId());
    TCarLocationContext context(dbTag.GetObjectId(), snapshot);
    const TCarLocationFeatures clf = context.GetCarAreaFeatures(eMode != EEvolutionMode::Default, server, TInstant::Zero());

    ITag::TAggregateEvolutionPolicy result;
    result.Policies.push_back(MakeAtomicShared<TScooterFixEvolutionPolicy>(clf));
    return result;
}

ITag::TAggregateEvolutionPolicy TBlockSessionEvolutionPolicyAction::BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode eMode, NDrive::TEntitySession& tx) const {
    if (!dbTag.Is<TChargableTag>()) {
        return {};
    }
    auto evlog = NDrive::GetThreadEventLogger();
    if (eMode != EEvolutionMode::Default) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "SkipBlockSessionEvolutionPolicy")
                ("evolution_mode", ToString(eMode))
            );
        }
        return {};
    }
    if (tx.GetOriginatorId()) {
        if (evlog) {
            evlog->AddEvent(NJson::TMapBuilder
                ("event", "SkipBlockSessionEvolutionPolicy")
                ("originator_id", tx.GetOriginatorId())
            );
        }
        return {};
    }
    auto&& fromTagName = dbTag->GetName();
    auto&& toTagName = Yensured(newTag)->GetName();
    if (!FromTagNames.contains(fromTagName)) {
        return {};
    }
    if (!ToTagNames.contains(toTagName)) {
        return {};
    }
    if (TraceTagNames.empty()) {
        return {};
    }
    auto optionalSession = server->GetDriveDatabase().GetSessionManager().GetTagSession(dbTag.GetTagId(), tx);
    R_ENSURE(optionalSession, {}, "cannot GetTagSession " << dbTag.GetTagId(), tx);
    auto session = *optionalSession;
    R_ENSURE(session, HTTP_INTERNAL_SERVER_ERROR, "cannot find session from tag " << dbTag.GetTagId(), tx);
    auto sessionId = session->GetSessionId();
    auto optionalTags = server->GetDriveDatabase().GetTagsManager().GetTraceTags().RestoreTagsRobust({ sessionId }, TraceTagNames, tx);
    R_ENSURE(optionalTags, {}, "cannot RestoreTags " << sessionId, tx);
    if (optionalTags->empty()) {
        return {};
    }
    auto&& tagName = Yensured(optionalTags->front())->GetName();
    return ITag::TAggregateEvolutionPolicy::CreateError(
        TStringBuilder() << "session tag " << tagName << " blocks evolution from " << fromTagName << " to " << toTagName
    ).SetUIErrorTitle(
        TStringBuilder() << "error.session_evolution_blocked_by_tag." << tagName << ".title"
    ).SetError(
        NDrive::MakeError("session_evolution_blocked_by_tag_" + tagName)
    ).SetErrorCode(HTTP_BAD_REQUEST);
}

bool TBlockSessionEvolutionPolicyAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& json) {
    return
        NJson::ParseField(json["from_tag_names"], FromTagNames) &&
        NJson::ParseField(json["to_tag_names"], ToTagNames) &&
        NJson::ParseField(json["trace_tag_names"], TraceTagNames);
}

NJson::TJsonValue TBlockSessionEvolutionPolicyAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue json;
    NJson::InsertField(json, "from_tag_names", FromTagNames);
    NJson::InsertField(json, "to_tag_names", ToTagNames);
    NJson::InsertField(json, "trace_tag_names", TraceTagNames);
    return json;
}

NDrive::TScheme TBlockSessionEvolutionPolicyAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    auto&& tagsMeta = Yensured(server)->GetDriveDatabase().GetTagsManager().GetTagsMeta();
    auto chargableTagNames = tagsMeta.GetRegisteredTagNames({ TChargableTag::TypeName });
    result.Add<TFSVariants>("from_tag_names").SetMultiSelect(true).SetVariants(chargableTagNames);
    result.Add<TFSVariants>("to_tag_names").SetMultiSelect(true).SetVariants(chargableTagNames);
    auto traceTags = tagsMeta.GetRegisteredTags(NEntityTagsManager::EEntityType::Trace);
    auto traceTagNames = NContainer::Keys(traceTags);
    result.Add<TFSVariants>("trace_tag_names").SetMultiSelect(true).SetVariants(traceTagNames);
    return result;
}
