#pragma once

#include <drive/backend/actions/abstract/action.h>
#include <drive/backend/chat/message.h>
#include <drive/backend/tags/tags.h>

class IEvolutionPolicyAction: public TUserAction {
private:
    using TBase = TUserAction;

public:
    using TBase::TBase;

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const = 0;
};

class TCovidVoluneteersEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TCovidVoluneteersEvolutionPolicyAction> Registrator;

private:
    R_READONLY(TString, TariffActionTag);
    R_READONLY(TString, PermitTagName);
    R_READONLY(TString, IgnoreTagName);
    R_READONLY(TString, PermitLandingId);
    R_READONLY(TString, WarningChatId);

public:
    using TBase::TBase;

    ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName();
    virtual TString GetType() const override;
};

class TRequiredPhotosEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "required_photos_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;
    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

private:
    R_FIELD(TVector<TString>, PhotoTagNames);
    R_FIELD(ui32, PhotoCount, 4);

private:
    static TFactory::TRegistrator<TRequiredPhotosEvolutionPolicyAction> Registrator;
};

class TSelfieVerificationEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TSelfieVerificationEvolutionPolicyAction> Registrator;

private:
    R_FIELD(TString, VerificationTagName);
    R_FIELD(TString, VerificationLandingId);
    R_FIELD(TString, RecheckDocumentsTagName);
    R_FIELD(TString, EventTagName);
    R_FIELD(TVector<TString>, TagsToAddOnSubmit);
    R_FIELD(bool, UseAntifraud, false);
    R_FIELD(bool, WaitForAntifraud, false);

public:
    using TBase::TBase;

    ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName();
    virtual TString GetType() const override;
};

class TRideCompletionEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TRideCompletionEvolutionPolicyAction> Registrator;

private:
    R_FIELD(bool, CheckIncorrectSignal, true);
    R_FIELD(bool, FreeDelegationEnabled, false);
    R_FIELD(TString, FreeDelegationConfirmationLanding);
    R_FIELD(TDuration, P2PDelegationDuration, TDuration::Minutes(3));
    R_FIELD(TDuration, ServiceDelegationDuration, TDuration::Hours(2));
    R_FIELD(TString, ServiceDelegationAllowedLanding);
    R_FIELD(TString, ServiceDelegationDeniedLanding);

public:
    using TBase::TBase;

    ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode /*eMode*/, NDrive::TEntitySession& /*session*/) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName();
    virtual TString GetType() const override;
};

class TTaxiRouteHistoryEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TTaxiRouteHistoryEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "taxi_route_history_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;
};

class TScooterFixPointEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TScooterFixPointEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "scooter_fix_point_evolution_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;
};

class TTaxisharingEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "taxisharing_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;
    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

private:
    R_FIELD(ui32, FuelLevelThreshold, 90);
    R_FIELD(TTagsFilter, OfferTagsFilter);

private:
    static TFactory::TRegistrator<TTaxisharingEvolutionPolicyAction> Registrator;
};

class TTelematicsCommandEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TTelematicsCommandEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "telematics_command_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

public:
    R_FIELD(NDrive::NVega::TCommand, Command);
    R_FIELD(TSet<TString>, OfferTags);
    R_FIELD(TString, From);
    R_FIELD(TString, To);
};

class TUserProximityEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TUserProximityEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "user_proximity_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& /*permissions*/, const NJson::TJsonValue& /*requestData*/, const EEvolutionMode /*eMode*/, NDrive::TEntitySession& /*session*/) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

private:
    double MaximalDistance = 1000;
    bool IgnoreUndefinedLocation = true;
};

class TCloseChatEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TCloseChatEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "close_chat_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

private:
    TVector<TString> ChatTagNames;
    TString MessageText;
    TTagsFilter OfferTagsFilter;
};

class TMoveChatEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TMoveChatEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "move_chat_policy";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }

    virtual ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

private:
    TString ChatName;
    TString NodeName;
    TString TargetTagName;
};

class TBlockSessionEvolutionPolicyAction: public IEvolutionPolicyAction {
private:
    using TBase = IEvolutionPolicyAction;

private:
    static TFactory::TRegistrator<TBlockSessionEvolutionPolicyAction> Registrator;

public:
    using TBase::TBase;

    static TString GetTypeName() {
        return "block_session_evolution_policy";
    }

    TString GetType() const override {
        return GetTypeName();
    }

    ITag::TAggregateEvolutionPolicy BuildEvolutionPolicy(const TConstDBTag& dbTag, ITag::TPtr newTag, const NDrive::IServer* server, const TUserPermissions& permissions, const NJson::TJsonValue& requestData, const EEvolutionMode eMode, NDrive::TEntitySession& session) const override;

    bool DeserializeSpecialsFromJson(const NJson::TJsonValue& json) override;
    NJson::TJsonValue SerializeSpecialsToJson() const override;
    NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

private:
    TSet<TString> FromTagNames;
    TSet<TString> ToTagNames;
    TSet<TString> TraceTagNames;
};
