#include "filter.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>

#include <rtline/library/json/adapters.h>
#include <rtline/library/json/parse.h>

TUserAction::TFactory::TRegistrator<TFilterAction> TFilterAction::Registrator("filter");

bool TFilterAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    ReportBase = jsonValue["report_base"];
    Text = jsonValue["text"].GetString();
    if (!Text) {
        return false;
    }
    Subtext = jsonValue["subtext"].GetString();
    Icon = jsonValue["icon"].GetString();
    Meta = jsonValue["meta"].GetBoolean();
    Priority = jsonValue["priority"].GetInteger();
    TJsonProcessor::ReadContainer(jsonValue, "excludes", Excludes);
    TJsonProcessor::ReadContainer(jsonValue, "parent_objects", ParentObjects);
    if (Excludes.contains(GetName())) {
        return false;
    }
    if (ParentObjects.contains(GetName())) {
        return false;
    }
    if (!NJson::ParseField(jsonValue["autoselect"], Autoselect)) {
        return false;
    }
    if (!NJson::ParseField(jsonValue["explicit"], Explicit)) {
        return false;
    }
    if (!NJson::ParseField(jsonValue["marker"], Marker)) {
        return false;
    }
    if (!NJson::ParseField(jsonValue["style"], Style)) {
        return false;
    }
    if (!NJson::ParseField(jsonValue["type"], FilterType)) {
        return false;
    }
    return TagsFilter.DeserializeFromJson(jsonValue);
}

NJson::TJsonValue TFilterAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TagsFilter.SerializeToJson();
    result["autoselect"] = Autoselect;
    result["text"] = Text;
    result["subtext"] = Subtext;
    result["icon"] = Icon;
    result["explicit"] = Explicit;
    result["marker"] = Marker;
    result["meta"] = Meta;
    result["priority"] = Priority;
    result["report_base"] = ReportBase;
    result["style"] = NJson::ToJson(Style);
    result["type"] = NJson::ToJson(FilterType);
    TJsonProcessor::WriteContainerArray(result, "excludes", Excludes, false);
    TJsonProcessor::WriteContainerArray(result, "parent_objects", ParentObjects, false);
    return result;
}

NJson::TJsonValue TFilterAction::GetReport(const TSet<TString>* allFilterIds) const {
    NJson::TJsonValue result = ReportBase;
    result["text"] = Text;
    result["icon"] = Icon;
    result["name"] = GetName();
    result["id"] = GetSequentialId();
    for (auto&& id : Excludes) {
        if (allFilterIds && !allFilterIds->contains(id)) {
            continue;
        }
        result["excludes"].AppendValue(id);
    }
    for (auto&& id : ParentObjects) {
        if (allFilterIds && !allFilterIds->contains(id)) {
            continue;
        }
        result["parent_objects"].AppendValue(id);
    }
    if (Autoselect) {
        result["autoselect"] = Autoselect;
    }
    if (Marker) {
        result["marker"] = Marker;
    }
    if (Meta) {
        result["meta"] = Meta;
    }
    if (Subtext) {
        result["subtext"] = Subtext;
    }
    if (Style) {
        result["style"] = NJson::ToJson(Style);
    }
    if (FilterType) {
        result["type"] = NJson::ToJson(FilterType);
    }
    return result;
}

NDrive::TScheme TFilterAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("conditions");
    result.Add<TFSString>("text", "name");
    result.Add<TFSString>("subtext", "Текст сереньким");
    result.Add<TFSString>("icon");
    result.Add<TFSString>("marker", "Marker (for DRIVEBACK-3275)");
    result.Add<TFSBoolean>("autoselect");
    result.Add<TFSBoolean>("explicit");
    result.Add<TFSBoolean>("meta", "Метафильтр");
    result.Add<TFSNumeric>("priority");
    result.Add<TFSJson>("report_base");
    TVector<TFilterAction> actions = server->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetActionsWithType<TFilterAction>();
    TVector<TString> parents;
    for (auto&& i : actions) {
        parents.emplace_back(i.GetName());
    }
    result.Add<TFSVariants>("excludes", "Взаимоисключаемые фильтры").SetVariants(parents).SetMultiSelect(true);
    result.Add<TFSVariants>("parent_objects", "Родительские объекты").SetVariants(parents).SetMultiSelect(true);
    result.Add<TFSVariants>("style", "Стиль фильтра-родителя").InitVariants<EStyle>();
    result.Add<TFSVariants>("type", "Тип фильтра").InitVariants<EType>();
    return result;
}

template <>
NJson::TJsonValue NJson::ToJson(const TFilterAction::EStyle& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TFilterAction::EStyle& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

template <>
NJson::TJsonValue NJson::ToJson(const TFilterAction::EType& object) {
    return NJson::ToJson(NJson::Stringify(object));
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TFilterAction::EType& result) {
    return NJson::TryFromJson(value, NJson::Stringify(result));
}

TCompositeFilter::TCompositeFilter(const TString& actionId, const NDrive::IServer* server) {
    TSet<TString> actionIds;
    StringSplitter(actionId).SplitBySet("* ").SkipEmpty().Collect(&actionIds);
    auto fetchedActions = server->GetDriveAPI()->GetRolesManager()->template GetActionsAs<TFilterAction>(actionIds);
    for (auto&& i : fetchedActions) {
        Actions.emplace_back(std::move(i.second));
    }
}
