#pragma once

#include <drive/backend/actions/abstract/action.h>

#include <drive/backend/tags/tags_filter.h>

#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/deprecated/split/delim_string_iter.h>

#include <util/string/subst.h>

class TFilterAction: public TUserAction {
private:
    using TBase = TUserAction;

public:
    enum class EStyle {
        Simple      /* "simple" */,
        Models      /* "models" */,
    };
    enum class EType {
        Checkbox    /* "checkbox" */,
        Period      /* "period" */,
        Radio       /* "radio" */,
        Switch      /* "switch" */,
    };

public:
    R_FIELD(TSet<TString>, Excludes);
    R_FIELD(TSet<TString>, ParentObjects);
    R_FIELD(TString, Text);
    R_FIELD(TString, Subtext);
    R_FIELD(TString, Icon);
    R_FIELD(TString, Marker);
    R_FIELD(NJson::TJsonValue, ReportBase);
    R_FIELD(ui32, Priority, Max<ui32>());
    R_FIELD(TMaybe<EStyle>, Style, {});
    R_FIELD(TMaybe<EType>, FilterType);
    R_FIELD(bool, Autoselect, false);
    R_FIELD(bool, Explicit, false);
    R_FIELD(bool, Meta, false);

private:
    static constexpr EStyle DefaultStyle = EStyle::Models;

    static TFactory::TRegistrator<TFilterAction> Registrator;

public:
    TFilterAction() = default;
    TFilterAction(const TString& name, const TString& tagsFilter)
        : TBase(name)
    {
        TagsFilter.DeserializeFromString(tagsFilter);
    }

    const TTagsFilter& GetTagsFilter() const {
        return TagsFilter;
    }

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    NJson::TJsonValue GetReport(const TSet<TString>* allFilterIds = nullptr) const;

    static TString GetTypeName() {
        return "filter";
    }

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual TString GetType() const override {
        return GetTypeName();
    }

    bool Accept(const TTaggedObject& object) const {
        return TagsFilter.IsMatching(object);
    }

    bool Accept(const TVector<TDBTag>& tags) const {
        return TagsFilter.IsMatching(tags);
    }

    bool operator<(const TFilterAction& other) const {
        return GetPriority() < other.GetPriority() || (GetPriority() == other.GetPriority() && GetName() < other.GetName());
    }

private:
    TTagsFilter TagsFilter;
};

class TCompositeFilter {
private:
    TVector<TFilterAction> Actions;

public:
    TCompositeFilter(const TString& actionId, const NDrive::IServer* server);

    bool Match(const TTaggedObject& t) const {
        for (auto&& i : Actions) {
            if (!i.Accept(t)) {
                return false;
            }
        }
        return true;
    }
};

class TFilterActionSet {
private:
    TVector<const TCompositeFilter*> Filters;

public:
    TFilterActionSet(TVector<const TCompositeFilter*>&& filters)
        : Filters(std::move(filters))
    {
    }

    bool Empty() const {
        return Filters.empty();
    }

    bool Match(const TTaggedDevice& device) const {
        if (Empty()) {
            return true;
        }
        for (auto&& filter : Filters) {
            if (filter->Match(device)) {
                return true;
            }
        }
        return false;
    }
};
