#include "info_access.h"

#include <drive/backend/abstract/base.h>
#include <drive/backend/billing/manager.h>
#include <drive/backend/chat/engine.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/doc_packages/manager.h>
#include <drive/backend/roles/manager.h>

TUserAction::TFactory::TRegistrator<TInfoAccessAction> TInfoAccessAction::Registrator(TInfoAccessAction::GetTypeName());

bool TInfoAccessAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& jsonValue) {
    DeviceTraits = NDeviceReport::NoTraits;
    UserTraits = NUserReport::NoTraits;
    CarRegistryTraits = NAttachmentReport::NoStructureTraits;
    MessageVisibilityTraits = 0;
    return
        NJson::ParseField(jsonValue["car_registry_report_traits"], NJson::BitMask<NRegistryDocumentReport::EReportTraits>(CarRegistryTraits)) &&
        NJson::ParseField(jsonValue["device_report_traits"], NJson::BitMask<NDeviceReport::EReportTraits>(DeviceTraits)) &&
        NJson::ParseField(jsonValue["device_search_traits"], NJson::BitMask<NDeviceReport::EReportTraits>(DeviceSearchTraits)) &&
        NJson::ParseField(jsonValue["user_report_traits"], NJson::BitMask<NUserReport::EReportTraits>(UserTraits)) &&
        NJson::ParseField(jsonValue["user_search_traits"], NJson::BitMask<NUserReport::EReportTraits>(UserSearchTraits)) &&
        NJson::ParseField(jsonValue["message_visibility_traits"], NJson::BitMask<NDrive::NChat::TMessage::EMessageTraits>(MessageVisibilityTraits)) &&
        NJson::ParseField(jsonValue["available_documents"], AvailableDocuments);
}

NJson::TJsonValue TInfoAccessAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = NJson::JSON_MAP;
    NJson::InsertField(result, "available_documents", AvailableDocuments);
    NJson::InsertField(result, "car_registry_report_traits", NJson::BitMask<NRegistryDocumentReport::EReportTraits>(CarRegistryTraits));
    NJson::InsertField(result, "device_report_traits", NJson::BitMask<NDeviceReport::EReportTraits>(DeviceTraits));
    NJson::InsertField(result, "device_search_traits", NJson::BitMask<NDeviceReport::EReportTraits>(DeviceSearchTraits));
    NJson::InsertField(result, "user_report_traits", NJson::BitMask<NUserReport::EReportTraits>(UserTraits));
    NJson::InsertField(result, "user_search_traits", NJson::BitMask<NUserReport::EReportTraits>(UserSearchTraits));
    NJson::InsertField(result, "message_visibility_traits", NJson::BitMask<NDrive::NChat::TMessage::EMessageTraits>(MessageVisibilityTraits));
    return result;
}

NDrive::TScheme TInfoAccessAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSVariants>("device_report_traits", "Информация для просмотра по машине").SetMultiSelect(true).InitVariants<NDeviceReport::EReportTraits>();
    result.Add<TFSVariants>("device_search_traits", "Информация для поиска по машинам").SetMultiSelect(true).InitVariants<NDeviceReport::EReportTraits>();
    result.Add<TFSVariants>("user_report_traits", "Информация для просмотра по пользователю").SetMultiSelect(true).InitVariants<NUserReport::EReportTraits>();
    result.Add<TFSVariants>("user_search_traits", "Информация для поиска по пользователям").SetMultiSelect(true).InitVariants<NUserReport::EReportTraits>();
    result.Add<TFSVariants>("car_registry_report_traits", "Информация для просмотра по реестру авто").SetMultiSelect(true).InitVariants<NRegistryDocumentReport::EReportTraits>();
    result.Add<TFSVariants>("message_visibility_traits", "Доступные для прочтения спец. типы сообщений").SetMultiSelect(true).InitVariants<NDrive::NChat::TMessage::EMessageTraits>();

    TVector<TDocumentDescriptionPtr> documentsPtr;
    if (!!server && server->GetDocumentsManager() && server->GetDocumentsManager()->GetRegisteredDocuments(documentsPtr)) {
        TSet<TString> names;
        for (const auto& doc : documentsPtr) {
            names.insert(doc.GetInternalId());
        }
        result.Add<TFSVariants>("available_documents", "Доступные к просмотру документы").SetMultiSelect(true).SetVariants(names);
    }

    if (!!server && !!server->GetDriveAPI() && server->GetDriveAPI()->HasBillingManager()) {
        TVector<TString> accounts;
        auto registeredAccounts = server->GetDriveAPI()->GetBillingManager().GetAccountsManager().GetRegisteredAccounts();
        for (auto&& description : registeredAccounts) {
            accounts.emplace_back(description.GetName());
        }
    }
    return result;
}
