#include "session_photo_screen.h"

#include <drive/backend/offers/actions/abstract.h>

namespace NDrive {

    const TString TSessionPhotoScreenAction::TypeName = "session_photo_screen";

    TUserAction::TFactory::TRegistrator<TSessionPhotoScreenAction> TSessionPhotoScreenAction::Registrator(TSessionPhotoScreenAction::TypeName);

    TString TSessionPhotoScreenAction::GetType() const {
        return TypeName;
    }

    NJson::TJsonValue TSessionPhotoScreenAction::SerializeSpecialsToJson() const {
        NJson::TJsonValue json = TBase::SerializeSpecialsToJson();
        NJson::InsertField(json, "last_segments", LastSegments);
        NJson::InsertField(json, "last_duration", LastDuration);
        NJson::InsertField(json, "minimal_last_duration", MinimalLastDuration);
        NJson::InsertField(json, "user_distance", UserDistance);
        NJson::InsertField(json, "offer_tags_filter", OfferTagsFilter);
        NJson::InsertField(json, "photo_required", PhotoRequired);
        NJson::InsertField(json, "photo_count", PhotoCount);
        return json;
    }

    bool TSessionPhotoScreenAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "last_segments", LastSegments, false) &&
            NJson::ParseField(json, "last_duration", LastDuration, false) &&
            NJson::ParseField(json, "minimal_last_duration", MinimalLastDuration, false) &&
            NJson::ParseField(json, "user_distance", UserDistance, false) &&
            NJson::ParseField(json, "offer_tags_filter", OfferTagsFilter) &&
            NJson::ParseField(json, "photo_required", PhotoRequired, false) &&
            NJson::ParseField(json, "photo_count", PhotoCount, false) &&
            TBase::DeserializeSpecialsFromJson(json);
    }

    NDrive::TScheme TSessionPhotoScreenAction::DoGetScheme(const NDrive::IServer* server) const {
        NDrive::TScheme scheme = TBase::DoGetScheme(server);
        TVector<TString> alowedTags = {
            TChargableTag::Prereservation,
            TChargableTag::Reservation,
            TChargableTag::Acceptance,
            TChargableTag::Riding,
            TChargableTag::Parking,
            TChargableTag::Servicing,
            TChargableTag::Sharing,
        };
        scheme.Add<TFSArray>("last_segments", "Применять после этапов").SetElement<TFSVariants>().SetVariants(alowedTags);
        scheme.Add<TFSNumeric>("last_duration", "Применять, если длительность этапа меньше");
        scheme.Add<TFSNumeric>("minimal_last_duration", "Применять, если длительность этапа больше");
        scheme.Add<TFSNumeric>("user_distance", "Применять, если расстояние пользователя меньше");
        scheme.Add<TFSString>("offer_tags_filter", "Фильтр тегов офферов");
        scheme.Add<TFSBoolean>("photo_required", "Требовать фотографии");
        scheme.Add<TFSNumeric>("photo_count", "Количество фотографий");
        return scheme;
    }

    bool TSessionPhotoScreenAction::CheckSession(
        const NDrive::IServer& server,
        const TUserPermissions& permissions,
        const TFullCompiledRiding& ride,
        const TMaybe<NDrive::TLocation>& carLocation
    ) const {
        TString lastSegmentName = TChargableTag::Reservation;
        TDuration lastSegmentDuration = ride.GetDuration();
        auto&& events = ride.GetLocalEvents();
        if (auto sz = events.size(); sz >= 2) {
            auto prevPos = sz - 2;
            if (sz >= 3 && events[prevPos].GetTagName() == TChargableTag::Reservation) {
                --prevPos;
            }
            for (i32 i = i32(sz) - 3; i >= 0; --i) {
                if (events[i].GetTagName() != events[prevPos].GetTagName()) {
                    break;
                }
                prevPos = i;
            }
            lastSegmentName = events[prevPos].GetTagName();
            lastSegmentDuration = ride.GetFinishInstant() - events[prevPos].GetInstant();
        }
        if (LastSegments && !LastSegments.contains(lastSegmentName)) {
            return false;
        }
        if (LastDuration && LastDuration < lastSegmentDuration) {
            return false;
        }
        if (MinimalLastDuration && MinimalLastDuration > lastSegmentDuration) {
            return false;
        }
        if (UserDistance) {
            auto userLocation = permissions.GetUserFeatures().GetUserLocationPtr();
            if (!userLocation || !carLocation) {
                return false;
            }
            auto distance = carLocation->GetCoord().GetLengthTo(*userLocation);
            if (UserDistance < distance) {
                return false;
            }
        }
        if (OfferTagsFilter) {
            auto offer = ride.GetOffer();
            auto action = offer ? server.GetDriveDatabase().GetUserPermissionManager().GetAction(offer->GetBehaviourConstructorId()) : Nothing();
            auto builder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
            if (!builder || !OfferTagsFilter.IsMatching(builder->GetGrouppingTags())) {
                return false;
            }
        }
        return true;
    }

}
