#include "tag.h"

#include <drive/backend/tags/tag_description.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/data/organization_tag.h>
#include <drive/backend/roles/permissions.h>

#include <rtline/util/json_processing.h>

#include <library/cpp/regex/pcre/regexp.h>

TUserAction::TFactory::TRegistrator<TTagAction> TTagAction::Registrator(TTagAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TObjectAccessAction> TObjectAccessAction::Registrator(TObjectAccessAction::GetTypeName());
TUserAction::TFactory::TRegistrator<TObjectSelectorUsageAction> TObjectSelectorUsageAction::Registrator(TObjectSelectorUsageAction::GetTypeName());

void TEntityTypesSelector::SerializeToJson(NJson::TJsonValue& value) const {
    JWRITE(value, "tag_attributes_filter", TagAttributesFilter.ToString());
}

bool TEntityTypesSelector::DeserializeFromJson(const NJson::TJsonValue& value) {
    if (value.Has("tag_attributes_filter")) {
        if (!value["tag_attributes_filter"].IsString()) {
            return false;
        }
        TagAttributesFilter.DeserializeFromString(value["tag_attributes_filter"].GetString());
    }
    return true;
}

ui32 TObjectSelectorAction::CalcHash(TStringBuf id) {
    ui16 hash = FnvHash<ui32>(id);
    return hash;
}

bool TObjectSelectorAction::Check(ui32 hash, TInstant timestamp) const {
    ui32 hashLocal = hash;
    if (SaltDiscretization != TDuration::Zero()) {
        hashLocal += ((int)(timestamp - SaltStart).Seconds() / SaltDiscretization.Seconds());
    }
    hashLocal = hashLocal % MaxHash;
    if (ObjectHashMin == ObjectHashMax && ObjectHashMax == 0) {
        return true;
    } else {
        return (ObjectHashMin <= hashLocal) && (hashLocal < ObjectHashMax);
    }
}

bool TObjectSelectorAction::Check(TStringBuf id, TInstant timestamp) const {
    auto hash = CalcHash(id);
    return Check(hash, timestamp);
}

NDrive::TScheme TObjectSelectorAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSNumeric>("device_hash_min", "'['Левая граница хешей объектов 0-" + ::ToString(MaxHash)).SetMin(0).SetMax(MaxHash).SetDefault(0);
    result.Add<TFSNumeric>("device_hash_max", "')'Правая граница хешей объектов 0-" + ::ToString(MaxHash)).SetMin(0).SetMax(MaxHash).SetDefault(0);

    result.Add<TFSNumeric>("salt_start", "Время начала отсчета коррекции хеша (c)").SetMin(0).SetMax(Max<ui32>()).SetDefault(TInstant::Hours(15).Seconds());
    result.Add<TFSDuration>("salt_discretization", "Дискретизация коррекции хеша").SetDefault(TDuration::Days(1));

    result.Add<TFSVariants>("actions").SetMultiSelect(true).InitVariants<EAction>();
    TEntityTypesSelector::CorrectScheme(result);
    return result;
}

NJson::TJsonValue TObjectSelectorAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    JWRITE_DEF(result, "device_hash_min", ObjectHashMin, 0);
    JWRITE_DEF(result, "device_hash_max", ObjectHashMax, 0);
    JWRITE_INSTANT(result, "salt_start", SaltStart);
    JWRITE_DURATION(result, "salt_discretization", SaltDiscretization);
    TJsonProcessor::WriteContainerArrayStrings(result, "actions", Actions);
    TEntityTypesSelector::SerializeToJson(result);
    return result;
}

bool TObjectSelectorAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DeserializeSpecialsFromJson(value)) {
        return false;
    }
    JREAD_INT_OPT(value, "device_hash_min", ObjectHashMin);
    if (ObjectHashMin > MaxHash) {
        return false;
    }
    JREAD_INT_OPT(value, "device_hash_max", ObjectHashMax);
    if (ObjectHashMax > MaxHash) {
        return false;
    }
    if (ObjectHashMin > ObjectHashMax) {
        return false;
    }
    JREAD_CONTAINER_OPT(value, "actions", Actions);
    JREAD_INSTANT_OPT(value, "salt_start", SaltStart);
    JREAD_DURATION_OPT(value, "salt_discretization", SaltDiscretization);
    if (!TEntityTypesSelector::DeserializeFromJson(value)) {
        return false;
    }
    return true;
}

bool TTagAction::HasCondition() const {
    return !FilterAllow.IsEmpty() || !FilterDeny.IsEmpty() || PhotosNecessaryCount || CheckAccount;
}

bool TTagAction::Match(const TString& tagName) const {
    if (Matcher) {
        return Matcher->Match(tagName.c_str());
    } else {
        return tagName == TagName;
    }
}

bool TTagAction::Match(const TTagDescription& tagDescription) const {
    return Match(tagDescription.GetName());
}

TTagAction::ECheckResult TTagAction::CheckObjectStatus(const TConstDBTag& tag, const TUserPermissions& permissions, const TVector<TDBTag>& tags, const TObjectEvents<TConstDBTag>& history, TString* error) const {
    if (!tag) {
        return ECheckResult::Problem;
    }
    if (!GetFilterDeny().IsEmpty()) {
        if (GetFilterDeny().IsMatching(tags, error)) {
            return ECheckResult::Deny;
        }
    }
    if (!GetFilterAllow().IsEmpty()) {
        if (!GetFilterAllow().IsMatching(tags, error)) {
            return ECheckResult::Ignore;
        }
    }
    if (auto organizationTag = tag.GetTagAs<IOrganizationTag>(); organizationTag && CheckAccount) {
        if (!permissions.GetActualAccounts().contains(organizationTag->GetParentId())) {
            if (!!error) {
                *error = "not_actual_account_for_tag_action";
            }
            return ECheckResult::Ignore;
        }
    }
    if (GetPhotosNecessaryCount()) {
        ui32 photosCount = 0;
        for (auto&& i : history) {
            if (i.GetHistoryAction() == EObjectHistoryAction::AddSnapshot && i && tag->GetName() == i->GetName()) {
                ++photosCount;
            }
        }
        if (photosCount < GetPhotosNecessaryCount()) {
            if (!!error) {
                *error = "not_enough_photos_for_tag_action";
            }
            return ECheckResult::Ignore;
        }
    }
    return ECheckResult::Accept;
}

bool TTagAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DeserializeSpecialsFromJson(value)) {
        return false;
    }
    ETagAction action;
    if (value["tag_action"].IsArray()) {
        for (auto&& i : value["tag_action"].GetArray()) {
            if (TryFromString(i.GetString(), action)) {
                TagActions.emplace(action);
            } else {
                WARNING_LOG << "Incorrect action for tag " << i.GetString() << Endl;
            }
        }
    } else if (TryFromString(value["tag_action"].GetString(), action)) {
        TagActions.emplace(action);
    } else {
        return false;
    }
    if (value.Has("filter_allow")) {
        if (!value["filter_allow"].IsString() || !FilterAllow.DeserializeFromString(value["filter_allow"].GetString())) {
            return false;
        }
    }
    if (value.Has("filter_deny")) {
        if (!value["filter_deny"].IsString() || !FilterDeny.DeserializeFromString(value["filter_deny"].GetString())) {
            return false;
        }
    }
    JREAD_INT_OPT(value, "photos_count", PhotosNecessaryCount);
    TagName = value["tag_name"].GetString();
    if (TagName.StartsWith("$") && TagName.size() > 1) {
        Matcher = MakeHolder<TRegExMatch>();
        Matcher->Compile(TagName.substr(1));
        if (!Matcher->IsCompiled()) {
            return false;
        }
    }
    if (!TEntityTypesSelector::DeserializeFromJson(value)) {
        return false;
    }
    JREAD_BOOL_OPT(value, "check_account", CheckAccount);
    return true;
}

NJson::TJsonValue TTagAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue tagActions = NJson::JSON_ARRAY;
    for (auto&& i : TagActions) {
        tagActions.AppendValue(ToString(i));
    }
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    if (!FilterAllow.IsEmpty()) {
        result.InsertValue("filter_allow", FilterAllow.ToString());
    }
    if (!FilterDeny.IsEmpty()) {
        result.InsertValue("filter_deny", FilterDeny.ToString());
    }
    JWRITE(result, "photos_count", PhotosNecessaryCount);
    result["tag_action"] = tagActions;
    result["tag_name"] = ToString(TagName);
    JWRITE(result, "check_account", CheckAccount);
    TEntityTypesSelector::SerializeToJson(result);
    return result;
}

NDrive::TScheme TTagAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSText>("tag_name");
    result.Add<TFSString>("filter_allow", "Фильтр допустимых объектов для данного действия");
    result.Add<TFSString>("filter_deny", "Фильтр НЕ допустимых объектов для данного действия");
    result.Add<TFSNumeric>("photos_count", "Необходимое количество фотографий");
    result.Add<TFSVariants>("tag_action", "Действия при работе с тегом").SetMultiSelect(true).InitVariants<ETagAction>();
    TEntityTypesSelector::CorrectScheme(result);
    result.Add<TFSBoolean>("check_account", "Проверять принадлежность к организации");
    return result;
}

bool TObjectAccessAction::IsMatching(const TVector<TDBTag>& tags) const {
    if (Filter.IsEmpty()) {
        if (NegativeFilter.IsEmpty()) {
            return true;
        } else {
            return !NegativeFilter.IsMatching(tags);
        }
    } else if (NegativeFilter.IsEmpty()) {
        return Filter.IsMatching(tags);
    } else {
        return Filter.IsMatching(tags) && !NegativeFilter.IsMatching(tags);
    }
}

bool TObjectAccessAction::DeserializeSpecialsFromJson(const NJson::TJsonValue& value) {
    if (!TBase::DeserializeSpecialsFromJson(value)) {
        return false;
    }
    {
        const NJson::TJsonValue* filterJson;
        if (!value.GetValuePointer("filter", &filterJson)) {
            return false;
        }

        if (filterJson->IsString()) {
            if (!Filter.DeserializeFromString(filterJson->GetString())) {
                return false;
            }
        } else {
            if (!Filter.DeserializeFromJson(*filterJson)) {
                return false;
            }
        }
    }

    if (value.Has("negative_filter")) {
        TString negativeFilterString;
        if (!value["negative_filter"].GetString(&negativeFilterString)) {
            return false;
        }
        if (!NegativeFilter.DeserializeFromString(negativeFilterString)) {
            return false;
        }
    }
    return true;
}

NJson::TJsonValue TObjectAccessAction::SerializeSpecialsToJson() const {
    NJson::TJsonValue result = TBase::SerializeSpecialsToJson();
    result.InsertValue("filter", Filter.ToString());
    result.InsertValue("negative_filter", NegativeFilter.ToString());
    JWRITE_DEF(result, "acceptance_filter", AcceptanceFilter, true);
    return result;
}

NDrive::TScheme TObjectAccessAction::DoGetScheme(const NDrive::IServer* server) const {
    NDrive::TScheme result = TBase::DoGetScheme(server);
    result.Add<TFSString>("filter", "Прямой фильтр");
    result.Add<TFSString>("negative_filter", "Фильтр с отрицанием");
    return result;
}

void TEntityTypesSelector::CorrectScheme(NDrive::TScheme& result) const {
    result.Add<TFSString>("tag_attributes_filter", "Атрибутный фильтр тегов");
}
