#pragma once

#include <drive/backend/actions/abstract/action.h>

#include <drive/backend/tags/abstract.h>
#include <drive/backend/tags/tags_filter.h>

class TRegExMatch;
class TTagDescription;

class TEntityTypesSelector {
private:
    R_FIELD(TTagsFilter, TagAttributesFilter);

public:
    void CorrectScheme(NDrive::TScheme& result) const;

    void SerializeToJson(NJson::TJsonValue& value) const;
    bool DeserializeFromJson(const NJson::TJsonValue& value);
};

class TObjectSelectorAction: public TUserAction, public TEntityTypesSelector {
public:
    enum class EAction {
        Observe = 1 /* "observe" */,
        Perform = 2 /* "perform" */
    };

private:
    static constexpr int MaxHash = 1000;

private:
    using TBase = TUserAction;

public:
    R_FIELD(ui32, ObjectHashMin, 0);
    R_FIELD(ui32, ObjectHashMax, 0);
    R_FIELD(TSet<EAction>, Actions);

    R_FIELD(TInstant, SaltStart, TInstant::Hours(15));
    R_FIELD(TDuration, SaltDiscretization, TDuration::Days(1));

public:
    using TBase::TBase;

    static ui32 CalcHash(TStringBuf id);
    bool Check(ui32 hash, TInstant timestamp) const;
    bool Check(TStringBuf id, TInstant timestamp) const;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;
};

class TObjectAccessAction: public TObjectSelectorAction {
private:
    using TBase = TObjectSelectorAction;

public:
    R_FIELD(bool, AcceptanceFilter, true);
    R_FIELD(TTagsFilter, Filter);
    R_FIELD(TTagsFilter, NegativeFilter);

private:
    static TFactory::TRegistrator<TObjectAccessAction> Registrator;

public:
    using TBase::TBase;

    bool IsMatching(const TVector<TDBTag>& tags) const;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    virtual NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    static TString GetTypeName() {
        return "object_access";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }
};

class TObjectSelectorUsageAction: public TObjectAccessAction {
private:
    static TFactory::TRegistrator<TObjectSelectorUsageAction> Registrator;

public:
    static TString GetTypeName() {
        return "objects_selector";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }
};

class IEntityTagsManager;

class TTagAction: public TUserAction, public TEntityTypesSelector {
private:
    using TBase = TUserAction;

public:
    using TTagActions = NTagActions::TTagActions;
    using ETagAction = NTagActions::ETagAction;

public:
    R_FIELD(TString, TagName);
    R_READONLY(ui32, PhotosNecessaryCount, 0);
    R_READONLY(TTagsFilter, FilterAllow);
    R_READONLY(TTagsFilter, FilterDeny);
    R_READONLY(bool, CheckAccount, false);

private:
    THolder<TRegExMatch> Matcher;
    TSet<ETagAction> TagActions;

private:
    static TFactory::TRegistrator<TTagAction> Registrator;

public:
    using TBase::TBase;

    bool HasCondition() const;
    bool Match(const TString& tagName) const;
    bool Match(const TTagDescription& tagDescription) const;

    enum class ECheckResult {
        Accept,
        Deny,
        Ignore,
        Problem
    };

    ECheckResult CheckObjectStatus(const TConstDBTag& tag, const TUserPermissions& permissions, const TVector<TDBTag>& tags, const TObjectEvents<TConstDBTag>& history, TString* error = nullptr) const;

    const TSet<ETagAction>& GetTagActions() const {
        return TagActions;
    }

    TTagAction& AddTagAction(const ETagAction action) {
        TagActions.emplace(action);
        return *this;
    }

    NDrive::TScheme DoGetScheme(const NDrive::IServer* server) const override;

    virtual bool DeserializeSpecialsFromJson(const NJson::TJsonValue& value) override;
    virtual NJson::TJsonValue SerializeSpecialsToJson() const override;

    static TString GetTypeName() {
        return "tag";
    }

    virtual TString GetType() const override {
        return GetTypeName();
    }
};
