#include "base.h"

#include <drive/backend/database/drive/url.h>

#include <util/string/subst.h>

#include <regex>

namespace {
    TString GetNow(ui32 timezone) {
        return ::ToString((ModelingNow() - TDuration::Hours(timezone)).Seconds());
    }

    TString GetToday(ui32 timezone, ui32 daysShift = 0) {
        return ::ToString((TInstant::Days(ModelingNow().Days() + daysShift) - TDuration::Hours(timezone)).Seconds());
    }
}

TString INotifierActionBase::PatchData(const TString& objectId, const TString& original, const NAlerts::TFetcherContext& context) const {
    return PatchData(objectId, original, context, Timezone);
}

TString INotifierActionBase::PatchData(const TString& original, const TMap<TString, TString>& params) {
    TString result = original;
    for (auto&& templateData : params) {
        SubstGlobal(result, "<" + templateData.first + ">", templateData.second);
        SubstGlobal(result, "\\u003C" + templateData.first + "\\u003E", templateData.second);
    }
    return result;
}

TString INotifierActionBase::PatchData(const TString& objectId, const TString& original, const NAlerts::TFetcherContext& context, ui32 timezone) {
    TString result = original;

    SubstGlobal(result, "_TsToday_", GetToday(timezone));
    SubstGlobal(result, "_TsNow_", GetNow(timezone));
    {
        std::regex regexp("_TsToday\\+(\\d+)_");
        std::match_results<TString::const_iterator> matched;
        TVector<ui32> shifts;
        TString::const_iterator begin = result.begin();
        while (std::regex_search(begin, result.end(), matched, regexp)) {
            if (matched.size() == 2) {
                shifts.push_back(FromString<ui32>(TString(matched[1].first, matched[1].second)));
            }
            begin = matched[0].second;
        }
        for (auto&& shift : shifts) {
            SubstGlobal(result, "_TsToday+" + ::ToString(shift) + "_", GetToday(timezone, shift));
        }
    }

    SubstGlobal(result, "_CarUrl_", NEscJ::EscapeJ<false>(TCarsharingUrl().CarInfo(objectId)));
    SubstGlobal(result, "_ClientUrl_", NEscJ::EscapeJ<false>(TCarsharingUrl().ClientInfo(objectId)));
    SubstGlobal(result, "_SessionUrl_", NEscJ::EscapeJ<false>(TCarsharingUrl().SessionPage(objectId)));

    SubstGlobal(result, "_ObjectId_", objectId);

    auto it = context.GetFetchedParameters().find(objectId);
    if (it != context.GetFetchedParameters().end()) {
        result = PatchData(result, it->second);
    }
    return result;
}


bool INotifierActionBase::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_BOOL_OPT(jsonInfo, "save_fetched", SaveFetchedData);
    JREAD_INT_OPT(jsonInfo, "timezone", Timezone);
    return true;
}

NJson::TJsonValue INotifierActionBase::SerializeToJson() const {
    NJson::TJsonValue result(NJson::JSON_MAP);
    JWRITE_DEF(result, "timezone", Timezone, 3);
    JWRITE(result, "action_type", GetActionType());
    JWRITE_DEF(result, "save_fetched", SaveFetchedData, false);
    return result;
}

NDrive::TScheme INotifierActionBase::GetScheme(const IServerBase& /*server*/) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("save_fetched", "Сохранять данные для подстановки").SetDefault(false);
    scheme.Add<TFSNumeric>("timezone", "Таймзона. GMT+X").SetDefault(3);
    return scheme;
}

INotifierActionBase::TPtr INotifierActionBase::BuildFromJson(const NJson::TJsonValue& json) {
    TString actionType = json["action_type"].GetString();
    THolder<INotifierActionBase> action = THolder(TFactory::Construct(actionType));
    if (!action) {
        return nullptr;
    }
    if(!action->DeserializeFromJson(json)) {
        return nullptr;
    }
    return action.Release();
}

void INotifierActionBase::DoApply(const TString& objectId, const NAlerts::TFetchedMap& parameters, NAlerts::TFetcherContext& context) {
    if (SaveFetchedData) {
        for (auto [iteratorName, value] : parameters) {
            if (value != NAlerts::IFetchedIterator::InvalidData()) {
                context.SaveIteratorData(objectId, iteratorName, value);
            }
        }
    }
}
