#include "call.h"

#include <drive/backend/data/support_tags.h>
#include <drive/backend/support_center/manager.h>
#include <drive/backend/support_center/yandex/client.h>
#include <drive/backend/user_options/user_setting.h>

#include <drive/library/cpp/staff/client.h>
#include <drive/library/cpp/staff/entry.h>

#include <rtline/util/json_processing.h>

INotifierActionBase::TFactory::TRegistrator<TCallAction> TCallAction::Registrator("call");

TString TCallAction::GetActionType() const {
    return "call";
}

bool TCallAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    bool shouldSkip = true;
    for (auto&& objectId : objectIds) {
        shouldSkip = true;
        if (context.IsFiltered(objectId)) {
            continue;
        }
        TUserPermissions::TPtr permissions = context.GetServer()->GetDriveAPI()->GetUserPermissions(objectId, TUserPermissionsFeatures());
        if (permissions) {
            auto internalQueues = GetCallQueues(context, InternalTags);
            if (!internalQueues.empty()) {
                auto agentId = GetPhoneNumber(permissions, context);
                if (agentId.Defined()) {
                    TMessagesCollector errors;
                    if (ProcessInternalCallCenter(*agentId, context, internalQueues, errors)) {
                        shouldSkip = false;
                    } else {
                        ERROR_LOG << "TCallAction: TCallCenterYandexClient failed for user " << objectId << " error: " << errors.GetStringReport() << Endl;
                    }
                }
            }
        } else {
            ERROR_LOG << "TCallAction: can't retrieve permissions for user " << objectId << Endl;
        }

        if (shouldSkip) {
            context.SkipObject(objectId);
        }
    }
    return true;
}

bool TCallAction::ProcessInternalCallCenter(const TString& agentId, NAlerts::TFetcherContext& context, const TVector<TString>& queues, TMessagesCollector& errors) const {
    if (!context.GetServer()->GetSupportCenterManager() || !context.GetServer()->GetSupportCenterManager()->GetCallCenterYandexClient()) {
        ERROR_LOG << "TCallAction: CallCenterYandexClient not configured" << Endl;
        return false;
    }
    auto callCenterClient = context.GetServer()->GetSupportCenterManager()->GetCallCenterYandexClient();
    switch (CallActionType) {
    case (ECallActionType::Connect):
        return callCenterClient->ConnectAgent(agentId, queues, errors, false);
    case (ECallActionType::Disconnect):
        return callCenterClient->DisconnectAgent(agentId, queues, errors);
    case (ECallActionType::Pause):
        return callCenterClient->ChangeStatus(agentId, queues, errors, true);
    case (ECallActionType::Resume):
        return callCenterClient->ChangeStatus(agentId, queues, errors, false);
    default:
        ERROR_LOG << "TCallAction: wrong action type " << CallActionType << Endl;
        return false;
    }
}

TMaybe<TString> TCallAction::GetPhoneNumber(TUserPermissions::TPtr permissions, NAlerts::TFetcherContext& context) const {
    TUserSetting userSetting(permissions->GetUserId());
    {
        NDrive::TInfoEntitySession session;
        auto agentId = userSetting.GetValueFromCache(AgentIdSettingTagName, AgentIdSettingFieldName, session);
        if (agentId) {
            return agentId.GetRef();
        } else {
            ERROR_LOG << "TCallAction: GetPhoneNumber: can't ger user setting " << agentId.GetError() << " " << session.GetStringReport() << Endl;
        }
    }
    if (context.GetServer()->GetDriveAPI()->HasStaffClient()) {
        const TStaffClient& staffClient = context.GetServer()->GetDriveAPI()->GetStaffClient();
        TVector<TStaffEntry> entries;
        TMessagesCollector errors;
        auto login = permissions->GetLogin();
        if (staffClient.GetUserData(TStaffEntrySelector(TStaffEntrySelector::EStaffEntryField::Username, ToString(login)), entries, errors, {TStaffEntry::EStaffEntryField::WorkPhone})) {
            if (entries.size() > 0) {
                TString phoneNumber = ToString(entries[0].GetWorkPhone());
                auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::Writable>();
                if (!userSetting.SetValue(AgentIdSettingTagName, AgentIdSettingFieldName, phoneNumber, context.GetRobotUserId(), session) || !session.Commit()) {
                    ERROR_LOG << "TCallAction: can't save agent id to tag: " << session.GetStringReport() << Endl;
                }
                return phoneNumber;
            } else {
                ERROR_LOG << "TCallAction: StaffClient: no phone number for user " << permissions->GetUserId() << Endl;
            }
        } else {
            ERROR_LOG << "TCallAction: StaffClient: error: " << errors.GetStringReport() << Endl;
        }
    } else {
        ERROR_LOG << "TCallAction: StaffClient not configured" << Endl;
    }
    return {};
}

TVector<TString> TCallAction::GetCallQueues(const NAlerts::TFetcherContext& context, const TVector<TString>& callTagNames) const {
    TVector<TString> queues;
    for (auto&& tagName : callTagNames) {
        auto tagDescription = context.GetServer()->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetDescriptionByName(tagName);
        if (!tagDescription) {
            continue;
        }
        auto callDescription = dynamic_cast<const TSupportPhoneCallTag::TDescription*>(tagDescription.Get());
        if (callDescription && callDescription->GetCallQueue()) {
            queues.push_back(callDescription->GetCallQueue());
        }
    }
    return queues;
}

bool TCallAction::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    if (!NJson::ParseField(jsonInfo, "internal_tags", InternalTags, false)
        || !NJson::ParseField(jsonInfo, "audiotele_tags", AudioteleTags, false)
        || !NJson::ParseField(jsonInfo, "nextcontact_tags", NextcontactTags, false)
        || !NJson::ParseField(jsonInfo, "call_action", NJson::Stringify(CallActionType), false)
        || !NJson::ParseField(jsonInfo, "agent_id_setting_field_name", AgentIdSettingFieldName, true)
        || !NJson::ParseField(jsonInfo, "agent_id_setting_tag_name", AgentIdSettingTagName, true)) {
        return false;
    }
    return INotifierActionBase::DeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TCallAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    NJson::InsertField(result, "call_action", ToString(CallActionType));
    NJson::InsertField(result, "internal_tags", InternalTags);
    NJson::InsertField(result, "audiotele_tags", AudioteleTags);
    NJson::InsertField(result, "nextcontact_tags", NextcontactTags);
    NJson::InsertField(result, "agent_id_setting_field_name", AgentIdSettingFieldName);
    NJson::InsertField(result, "agent_id_setting_tag_name", AgentIdSettingTagName);
    return result;
}

NDrive::TScheme TCallAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    TSet<TString> tagNames;
    auto impl = server.GetAs<NDrive::IServer>();
    if (impl) {
        tagNames = impl->GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTagNames({TSupportPhoneCallTag::TypeName});
    }
    scheme.Add<TFSVariants>("call_action", "Действие с линией звонков").SetMultiSelect(false).InitVariants<TCallAction::ECallActionType>();
    scheme.Add<TFSVariants>("internal_tags", "Теги очередей внутреннего КЦ").SetMultiSelect(true).SetVariants(tagNames).SetEditable(true);
    scheme.Add<TFSVariants>("audiotele_tags", "Теги очередей КЦ audiotele").SetMultiSelect(true).SetVariants(tagNames).SetEditable(true);
    scheme.Add<TFSVariants>("nextcontact_tags", "Теги очередей КЦ nextcontact").SetMultiSelect(true).SetVariants(tagNames).SetEditable(true);
    scheme.Add<TFSString>("agent_id_setting_tag_name", "Тег с ID агента").SetRequired(true);
    scheme.Add<TFSString>("agent_id_setting_field_name", "Поле с ID агента").SetRequired(true);
    return scheme;
}
