#include "car_info_tag.h"

#include <drive/backend/alerts/actions/tag.h>

INotifierActionBase::TFactory::TRegistrator<TCarInfoObjectTagAction> TCarInfoObjectTagAction::Registrator("car_info_tag");

bool TCarInfoObjectTagAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    if (objectIds.empty()) {
        return true;
    }

    const IEntityTagsManager& entityTagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(ObjectType);
    const IDriveTagsManager& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager();

    TMessagesCollector errors;
    TTagActionImpl::TModifications modifications(context);
    modifications.SetAddTagPolicy(EUniquePolicy::NoUnique);
    for (auto&& carId : objectIds) {
        TString patchedTagData = INotifierActionBase::PatchData(carId, TagData, context);
        ITag::TPtr tagToAdd = IJsonSerializableTag::BuildFromString(tagsManager, patchedTagData, &errors);
        if (!tagToAdd) {
            context.AddError("CarInfoObjectTagAction", errors.GetStringReport());
            return false;
        }
        modifications.AddTag(ObjectId, tagToAdd);
    }
    return modifications.ApplyModification(entityTagsManager);
}

bool TCarInfoObjectTagAction::DeserializeFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "tag_data", TagData, true)
            && NJson::ParseField(json, "object_id", ObjectId, true)
            && NJson::ParseField(json, "object_type", NJson::Stringify(ObjectType), true)
            && INotifierActionBase::DeserializeFromJson(json);
}

NJson::TJsonValue TCarInfoObjectTagAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    NJson::InsertField(result, "tag_data", TagData);
    NJson::InsertField(result, "object_id", ObjectId);
    NJson::InsertField(result, "object_type", NJson::Stringify(ObjectType));
    return result;
}

NDrive::TScheme TCarInfoObjectTagAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSString>("object_id", "Id объекта, на который навесится тег");
    scheme.Add<TFSVariants>("object_type", "Тип объекта").InitVariants<NEntityTagsManager::EEntityType>();
    scheme.Add<TFSText>("tag_data", "Навешиваемый тег (_CarUrl_ для ссылки на машину)");
    return scheme;
}

TString TCarInfoObjectTagAction::GetActionType() const {
    return "car_info_tag";
}
