#include "chat.h"

#include <drive/backend/chat_robots/abstract.h>


INotifierActionBase::TFactory::TRegistrator<TChatAction> TChatAction::Registrator("user_chat");
INotifierActionBase::TFactory::TRegistrator<TChatNodeAction> TChatNodeAction::Registrator("chat_node");

TString TChatAction::GetActionType() const {
    return "user_chat";
}

bool TChatAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    auto chatRobot = context.GetServer()->GetChatRobot(ChatRobot);
    if (!chatRobot) {
        context.AddError("TChatAction", ChatRobot + " chat robot not configured");
        return false;
    }
    auto session = chatRobot->BuildChatEngineSession();
    for (auto&& objectId : objectIds) {
        if (!context.IsFiltered(objectId)) {
            TString text = PatchData(objectId, GetMessageTemplate(), context);
            NDrive::NChat::TMessage message;
            message.SetType(MessageType);
            message.SetText(text);
            if (!chatRobot->SendArbitraryMessage(objectId, Topic, context.GetRobotUserId(), message, session)) {
                context.AddError("TChatAction", "Cannot send message to " + objectId + ", error: " + session.GetStringReport());
                return false;
            }
        }
    }

    if (!session.Commit()) {
        context.AddError("TChatAction", "TChatAction сannot commit session " + session.GetStringReport());
        return false;
    }
    return true;
}

bool TChatAction::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_FROM_STRING(jsonInfo, "message_type", MessageType);
    JREAD_STRING_OPT(jsonInfo, "message_template", MessageTemplate);
    JREAD_STRING(jsonInfo, "chat_robot", ChatRobot);
    JREAD_STRING_OPT(jsonInfo, "chat_topic", Topic);
    return INotifierActionBase::DeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TChatAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE(result, "message_type", ::ToString(MessageType));
    JWRITE(result, "message_template", MessageTemplate);
    JWRITE(result, "chat_robot", ChatRobot);
    JWRITE(result, "chat_topic", Topic);
    return result;
}

NDrive::TScheme TChatAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSVariants>("message_type", "Тип сообщения").InitVariants<NDrive::NChat::TMessage::EMessageType>();
    scheme.Add<TFSString>("message_template", "Шаблон сообщения");

    auto robots = server.GetAs<NDrive::IServer>()->GetChatRobots();
    if (robots) {
        scheme.Add<TFSVariants>("chat_robot", "Тип робота").SetVariants(NContainer::Keys(robots));
    } else {
        scheme.Add<TFSVariants>("chat_robot", "Тип робота");
    }
    scheme.Add<TFSString>("chat_topic", "Топик");
    scheme.Add<TFSString>("robot_user_id", "RobotUserId");
    return scheme;
}

bool TChatNodeAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    auto chatRobot = context.GetServer()->GetChatRobot(ChatRobot);
    if (!chatRobot) {
        context.AddError("TChatNodeAction", ChatRobot + " chat robot not configured");
        return false;
    }
    auto session = chatRobot->BuildChatEngineSession();
    for (auto&& objectId : objectIds) {
        if (!context.IsFiltered(objectId)) {
            if (!chatRobot->MoveToStep(objectId, Topic, NodeId, &session, SendMessages)) {
                context.AddError("TChatNodeAction", "Cannot move chat to node " + objectId + ", error: " + session.GetStringReport());
                return false;
            }
        }
    }

    if (!session.Commit()) {
        context.AddError("TChatNodeAction", "TChatNodeAction сannot commit session " + session.GetStringReport());
        return false;
    }
    return true;
}

TString TChatNodeAction::GetActionType() const {
    return "chat_node";
}

bool TChatNodeAction::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    return
        INotifierActionBase::DeserializeFromJson(jsonInfo) &&
        NJson::ParseField(jsonInfo, "node_id", NodeId) &&
        NJson::ParseField(jsonInfo, "chat_robot", ChatRobot) &&
        NJson::ParseField(jsonInfo, "chat_topic", Topic) &&
        NJson::ParseField(jsonInfo, "send_messages", SendMessages, false);
}

NJson::TJsonValue TChatNodeAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    NJson::InsertField(result, "node_id", NodeId);
    NJson::InsertField(result, "chat_robot", ChatRobot);
    NJson::InsertField(result, "chat_topic", Topic);
    NJson::InsertField(result, "send_messages", SendMessages);
    return result;
}

NDrive::TScheme TChatNodeAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSString>("node_id", "ID ноды");
    scheme.Add<TFSBoolean>("send_messages", "Отправлять pre action messages").SetDefault(true);

    auto robots = server.GetAs<NDrive::IServer>()->GetChatRobots();
    if (robots) {
        scheme.Add<TFSVariants>("chat_robot", "Тип робота").SetVariants(NContainer::Keys(robots));
    } else {
        scheme.Add<TFSVariants>("chat_robot", "Тип робота");
    }
    scheme.Add<TFSString>("chat_topic", "Топик");
    return scheme;
}
