#pragma once

#include "base.h"
#include "tag.h"

#include <drive/backend/data/markers.h>

#include <rtline/library/unistat/cache.h>

#include <util/generic/algorithm.h>

class TChainNodeAction : public INotifierActionBase {
    using TModifications = TTagActionImpl::TModifications;
public:
    enum ENodeType {
        Control /* "control" */,
        Update /* "update" */,
        Add /* "add" */,
        Remove /* "remove" */,
    };
private:
    static TFactory::TRegistrator<TChainNodeAction> Registrator;
    NAlerts::TChainContext ChainContext;
    TVector<TString> AdditionalTags;
    ENodeType NodeType = ENodeType::Update;

public:
    virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
    virtual NJson::TJsonValue SerializeToJson() const override;
    virtual NDrive::TScheme GetScheme(const IServerBase& server) const override;
    virtual TString GetActionType() const override;

public:
    bool DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) override {
        return FinishImpl(objectIds, context, context.GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags());
    }

private:
    template <class TContainer>
    bool AddAdditionalTags(const TContainer& objectIds, const NAlerts::TFetcherContext& context, TModifications& modifications) const {
        const IDriveTagsManager& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager();
        for (auto&& object : objectIds) {
            for (auto tagJson : AdditionalTags) {
                auto patched = PatchData(object, tagJson, context);
                TMessagesCollector errors;
                ITag::TPtr tag = IJsonSerializableTag::BuildFromString(tagsManager, patched, &errors);
                if (!tag) {
                    ERROR_LOG << errors.GetStringReport() << Endl;
                    return false;
                }
                modifications.AddTag(object, tag);
            }
        }
        return true;
    }

    template <class TTagsManager>
    bool FinishImpl(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context, const TTagsManager& entityTagsManager) const {
        const NAlerts::TChainContext& chainContext = context.HasChainContext() ? context.GetChainContextUnsafe() : ChainContext;

        TModifications modifications(context);
        modifications.SetAddTagPolicy(EUniquePolicy::Undefined);
        if (!chainContext.GetStateTagName()) {
            if (!AddAdditionalTags(objectIds, context, modifications)) {
                context.AddError("TChainNodeAction", "Cannot add tags");
                return false;
            }
            return modifications.ApplyModification([](){}, entityTagsManager, false, "TChainNodeAction : " + chainContext.GetStateTagName());
        }

        ITag::TPtr tag = ITag::TFactory::Construct(TUserAlertTag::TypeName);
        if (!tag) {
            context.AddError("TChainNodeAction", "Cannot construct tag " + TUserAlertTag::TypeName);
            return false;
        }
        tag->SetName(chainContext.GetStateTagName());
        TAlertTag* newTagData = dynamic_cast<TAlertTag*>(tag.Get());
        newTagData->AddHistoryState(context.GetCurrentState(), context.GetFetchInstant());

        TDBTags objectTags;
        {
            auto session = entityTagsManager.BuildSession(true);
            auto optionalTags = entityTagsManager.RestoreTags(TVector<TString>{}, { chainContext.GetStateTagName() }, session);
            if (!optionalTags) {
                context.AddError("TChainNodeAction", "cannot RestoreTags " + session.GetStringReport());
                return false;
            }
            objectTags = std::move(*optionalTags);
        }

        ui32 toRemove = 0;
        TSet<TString> rejectedObjects;
        TSet<TString> withControlTag;
        TVector<TString> stateUpdates;
        {
            for (auto&& commonTag : objectTags) {
                if (commonTag->GetName() != chainContext.GetStateTagName()) {
                    continue;
                }

                const TAlertTag* tagData = commonTag.template GetTagAs<TAlertTag>();
                CHECK_WITH_LOG(tagData);

                const auto& objectId = commonTag.GetObjectId();
                withControlTag.emplace(objectId);

                if (!objectIds.contains(objectId)) {
                    if (NodeType == ENodeType::Control) {
                        modifications.RemoveTag(objectId, commonTag);
                        ++toRemove;
                    }
                } else {
                    if (chainContext.CheckState(tagData->GetStatesHistory(), tagData->GetStatesTs(), context.GetFetchInstant())) {
                        if (NodeType == ENodeType::Update) {
                            TDBTag tagCopy = commonTag.Clone(context.GetServer()->GetDriveAPI()->GetTagsHistoryContext());
                            TAlertTag* tagCopyData = tagCopy.MutableTagAs<TAlertTag>();
                            CHECK_WITH_LOG(tagCopyData);
                            tagCopyData->AddHistoryState(context.GetCurrentState(), context.GetFetchInstant());
                            modifications.UpdateTag(objectId, tagCopy);
                            stateUpdates.push_back(objectId);
                        } else if (NodeType == ENodeType::Remove) {
                            modifications.RemoveTag(objectId, commonTag);
                            ++toRemove;
                        }
                    } else {
                        rejectedObjects.insert(objectId);
                        context.SkipObject(objectId);
                    }
                }
            }
        }

        TVector<TString> filteredObjects(objectIds.size());
        {
            auto it = SetDifference(objectIds.begin(), objectIds.end(), rejectedObjects.begin(), rejectedObjects.end(), filteredObjects.begin());
            filteredObjects.resize(it - filteredObjects.begin());
        }

        if (NodeType == ENodeType::Control || NodeType == ENodeType::Add) {
            TVector<TString> finalTags;
            finalTags.reserve(filteredObjects.size());
            for (auto&& object : filteredObjects) {
                if (withControlTag.contains(object)) {
                    continue;
                }
                modifications.AddTag(object, tag);
                finalTags.push_back(object);
            }

            if (!AddAdditionalTags(finalTags, context, modifications)) {
                context.AddError("TChainNodeAction", "Cannot add final tags");
                return false;
            }
        } else {
            if (!AddAdditionalTags(stateUpdates, context, modifications)) {
                context.AddError("TChainNodeAction", "Cannot add update tags");
                return false;
            }
        }

        if (!modifications.ApplyModification([](){}, entityTagsManager, false, "TChainNodeAction : " + chainContext.GetStateTagName())) {
            context.AddError("TChainNodeAction", "Cannot apply modification");
            return false;
        }
        TUnistatSignalsCache::SignalAdd("robot_chain_node-" + context.GetCurrentState(), "removed", toRemove);
        TUnistatSignalsCache::SignalAdd("robot_chain_node-" + context.GetCurrentState(), "added", filteredObjects.size());
        TUnistatSignalsCache::SignalAdd("robot_chain_node-" + context.GetCurrentState(), "updated", stateUpdates.size());
        return true;
    }
};
