#include "role.h"

#include <drive/backend/alerts/fetchers/user_sessions.h>

#include <drive/backend/billing/manager.h>
#include <drive/backend/roles/manager.h>

#include <mapreduce/yt/interface/client.h>

INotifierActionBase::TFactory::TRegistrator<TUserRoleAction> TUserRoleAction::Registrator("user_roles");

bool TUserRoleAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    const TDriveAPI& driveApi = *context.GetServer()->GetDriveAPI();

    TSet<TString> otherUsersWithRoles;
    if (!driveApi.GetUsersData()->GetRoles().GetUsersWithAtLeastOneRoles(RoleNames, otherUsersWithRoles, false)) {
        context.AddError("TUserRoleAction", "Cannot get users with roles");
        return false;
    }
    for (auto it = otherUsersWithRoles.begin(); it != otherUsersWithRoles.end();) {
        if (objectIds.contains(*it)) {
            it = otherUsersWithRoles.erase(it);
        } else {
            ++it;
        }
    }

    auto session = driveApi.BuildTx<NSQL::Writable>();

    const TSet<TString>& forAdd = !Reverse ? objectIds : (AutoRevert ? otherUsersWithRoles : Default<TSet<TString>>());
    const TSet<TString>& forRemove = !Reverse ? (AutoRevert ? otherUsersWithRoles : Default<TSet<TString>>()) : objectIds;
    for (auto&& userId : forAdd) {
        auto optionalRoles = driveApi.GetUsersData()->GetRoles().RestoreUserRoles(userId, session);
        if (!optionalRoles) {
            context.AddError("UserRoleAction", "Cannot RestoreUserRoles: " + session.GetStringReport());
            return false;
        }
        TSet<TString> userRoleIds;
        for (const auto& role : optionalRoles->GetRoles()) {
            userRoleIds.emplace(role.GetRoleId());
            if (!OnlyAdd && RoleNames.contains(role.GetRoleId()) && !role.GetActive()) {
                TUserRole newRole = role;
                newRole.SetActive(true);
                if (!driveApi.GetRolesManager()->UpsertRoleForUser(newRole, context.GetRobotUserId(), session)) {
                    context.AddError("TUserRoleAction", "Cannot upsert role for user " + session.GetStringReport());
                    return false;
                }
            }
        }
        if (!OnlyActivate) {
            for (const auto& roleName : RoleNames) {
                if (!userRoleIds.contains(roleName)) {
                    TUserRole newRole;
                    newRole.SetUserId(userId);
                    newRole.SetRoleId(roleName);
                    if (!driveApi.GetRolesManager()->UpsertRoleForUser(newRole, context.GetRobotUserId(), session)) {
                        context.AddError("TUserRoleAction", "Cannot upsert role for user " + session.GetStringReport());
                        return false;
                    }
                }
            }
        }
    }

    for (auto&& userId : forRemove) {
        if (OnlyActivate && !OnlyAdd) {
            auto optionalRoles = driveApi.GetUsersData()->GetRoles().RestoreUserRoles(userId, session);
            if (!optionalRoles) {
                context.AddError("UserRoleAction", "Cannot RestoreUserRoles: " + session.GetStringReport());
                return false;
            }
            for (const auto& role : optionalRoles->GetRoles()) {
                if (RoleNames.contains(role.GetRoleId()) && role.GetActive()) {
                    TUserRole newRole = role;
                    newRole.SetActive(false);
                    if (!driveApi.GetRolesManager()->UpsertRoleForUser(newRole, context.GetRobotUserId(), session)) {
                        context.AddError("TUserRoleAction", "Cannot upsert role for user " + session.GetStringReport());
                        return false;
                    }
                }
            }
        } else {
            if (!driveApi.GetRolesManager()->RemoveRolesForUser(userId, RoleNames, context.GetRobotUserId(), session)) {
                context.AddError("TUserRoleAction", "Cannot remove role for user " + session.GetStringReport());
                return false;
            }
        }
    }

    if (!session.Commit()) {
        context.AddError("TUserRoleAction", "Cannot commit session " + session.GetStringReport());
        return false;
    }
    return true;
}

bool TUserRoleAction::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_BOOL_OPT(json, "auto_remove", AutoRevert); // deprecated
    JREAD_BOOL_OPT(json, "auto_revert", AutoRevert);
    TJsonProcessor::ReadContainer(json, "role_name", RoleNames, true);
    JREAD_BOOL_OPT(json, "only_activate", OnlyActivate);
    JREAD_BOOL_OPT(json, "only_add", OnlyAdd);
    JREAD_BOOL_OPT(json, "reverse", Reverse);
    if (Reverse && !OnlyActivate) {
        return false;
    }
    return true;
}

NJson::TJsonValue TUserRoleAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE_DEF(result, "auto_revert", AutoRevert, true);
    JWRITE_DEF(result, "auto_remove", AutoRevert, true); // deprecated
    JWRITE(result, "role_name", JoinSeq(",", RoleNames));
    JWRITE(result, "only_activate", OnlyActivate);
    JWRITE(result, "only_add", OnlyAdd);
    JWRITE_DEF(result, "reverse", Reverse, false);
    return result;
}

NDrive::TScheme TUserRoleAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSString>("role_name", "Роли");
    scheme.Add<TFSBoolean>("auto_revert", "Включить автоудаление ролей");
    scheme.Add<TFSBoolean>("only_activate", "Только активация");
    scheme.Add<TFSBoolean>("only_add", "Только навешивание");
    scheme.Add<TFSBoolean>("reverse", "Обратить действие");
    return scheme;
}

TString TUserRoleAction::GetActionType() const {
    return "user_roles";
}


INotifierActionBase::TFactory::TRegistrator<TWalletAction> TWalletAction::Registrator("manage_wallet");

bool TWalletAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    const TDriveAPI& driveApi = *context.GetServer()->GetDriveAPI();
    const TBillingManager& bManager = driveApi.GetBillingManager();

    auto session = driveApi.BuildTx<NSQL::Writable>();

    auto description = bManager.GetAccountsManager().GetDescriptionByName(AccountName);
    if (!description) {
        context.AddError("TWalletAction", "Incorrect wallet " + AccountName);
        return false;
    }
    NSQL::TQueryOptions linkOptions;
    linkOptions.AddGenericCondition("type_id", ToString(description->GetId()));
    auto accountLink = bManager.GetAccountsManager().GetLinksManager().GetLinkAccounts(std::move(linkOptions), session);
    if (!accountLink) {
        context.AddError("TWalletAction", session.GetStringReport());
        return false;
    }

    TMap<ui64, TSet<TString>> userAccounts;
    ForEach(accountLink->begin(), accountLink->end(), [&userAccounts](const auto& link) {
        userAccounts[link.GetAccountId()].emplace(link.GetUserId());
    });

    NSQL::TQueryOptions options;
    options.SetGenericCondition("account_id", MakeSet(NContainer::Keys(userAccounts)));
    auto accounts = bManager.GetAccountsManager().GetAccounts(std::move(options), session);
    if (!accounts) {
        context.AddError("TWalletAction", session.GetStringReport());
        return false;
    }
    TMap<TString, NDrive::NBilling::TAccountData> accountsByUser;

    TVector<ui32> disableAccounts;
    TVector<ui32> enableAccounts;
    for (auto&& account : *accounts) {
        auto users = userAccounts.Value(account.GetAccountId(), TSet<TString>());
        for (const auto& user : users) {
            if (objectIds.contains(user)) {
                accountsByUser[user] = account;
            } else {
                if (account.GetRecord()->IsActive()) {
                    disableAccounts.emplace_back(account.GetAccountId());
                }
            }
        }
    }

    TVector<TString> newUsers;
    for (auto&& userId : objectIds) {
        auto it = accountsByUser.find(userId);
        if (it != accountsByUser.end()) {
            if (!it->second.GetRecord()->IsActive()) {
                enableAccounts.emplace_back(it->second.GetAccountId());
            }
        } else {
            newUsers.emplace_back(userId);
        }
    }

    session.SetComment("TWalletAction");
    if (!bManager.GetAccountsManager().ActivateAccounts(disableAccounts, false, context.GetRobotUserId(), session)) {
        context.AddError("TWalletAction", "Cannot deactivate account " + session.GetStringReport());
        return false;
    }
    if (!bManager.GetAccountsManager().ActivateAccounts(enableAccounts, true, context.GetRobotUserId(), session)) {
        context.AddError("TWalletAction", "Cannot activate account " + session.GetStringReport());
        return false;
    }

    for (auto&& userId : newUsers) {
        NDrive::NBilling::TAccountRecord::TPtr accountRecord = NDrive::NBilling::TAccountRecord::TFactory::Construct(description->GetDataType());

        accountRecord->SetTypeId(description->GetId());
        accountRecord->SetActive(true);

        ui32 accountId = 0;
        if (!bManager.GetAccountsManager().RegisterAccount(accountRecord, context.GetRobotUserId(), session, &accountId)) {
            context.AddError("TWalletAction", "Cannot register account " + session.GetStringReport());
            return false;
        }

        auto account = bManager.GetAccountsManager().LinkAccount(userId, accountId, context.GetRobotUserId(), session);
        if (!account) {
            context.AddError("TWalletAction", "Cannot link account " + session.GetStringReport());
            return false;
        }
    }
    if (!session.Commit()) {
        context.AddError("TWalletAction", "TWalletAction Cannot commit session " + session.GetStringReport());
        return false;
    }
    return true;
}

bool TWalletAction::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING(json, "account_name", AccountName);
    return true;
}

NJson::TJsonValue TWalletAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE(result, "account_name", AccountName);
    return result;
}

NDrive::TScheme TWalletAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSString>("account_name", "Включить кошелек");
    return scheme;
}

TString TWalletAction::GetActionType() const {
    return "manage_wallet";
}

INotifierActionBase::TFactory::TRegistrator<TYtDumpAction> TYtDumpAction::Registrator("yt_dump");

bool TYtDumpAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    auto tagEntity = NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
    if (!tagEntity.Defined()) {
        context.AddError("TYtDumpAction", "Cannot get entity type");
        return false;
    }
    const IEntityTagsManager& entityTagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
    TVector<TDBTag> taggedObjects;
    {
        auto session = entityTagsManager.BuildSession(true);
        if (!entityTagsManager.RestoreTags({}, { TagName }, taggedObjects, session)) {
            context.AddError("TYtDumpAction", "Errors reading tags " + session.GetStringReport());
            return false;
        }
    }
    try {
        NYT::IClientPtr ytClient = NYT::CreateClient("hahn");
        auto transaction = ytClient->StartTransaction();
        if (!transaction) {
            context.AddError("TYtDumpAction", "Cannot start yt transaction");
            return false;
        }
        auto richYPath = NYT::TRichYPath(YTDir);
        auto writer = transaction->CreateTableWriter<NYT::TNode>(richYPath);

        for (auto&& dbTag : taggedObjects) {
            if (objectIds.contains(dbTag.GetObjectId())) {
                NYT::TNode recordNode;
                recordNode["user_id"] = dbTag.GetObjectId();
                recordNode["login"] = dbTag->GetComment();
                writer->AddRow(recordNode);
            }
        }
        writer->Finish();
        transaction->Commit();
    } catch (const std::exception& e) {
        context.AddError("TYtDumpAction", "YTError " + FormatExc(e));
        return false;
    }
    return true;
}

bool TYtDumpAction::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING(json, "tag_name", TagName);
    JREAD_STRING(json, "yt_dir", YTDir);
    return true;
}

NJson::TJsonValue TYtDumpAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE(result, "tag_name", TagName);
    JWRITE(result, "yt_dir", YTDir);
    return result;
}

NDrive::TScheme TYtDumpAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSString>("tag_name", "Тег");
    scheme.Add<TFSString>("yt_dir", "Таблица");
    return scheme;
}

TString TYtDumpAction::GetActionType() const {
    return "yt_dump";
}

//INotifierActionBase::TFactory::TRegistrator<TEvolveSessionAction> TEvolveSessionAction::Registrator("evolve_session");

bool TEvolveSessionAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    auto it = context.GetDataFetchers().find(NAlerts::EDataFetcherType::ESessions);
    if (it == context.GetDataFetchers().end()) {
        context.AddError("TEvolveSessionAction", "Cannot get data fetcher");
        return false;
    }

    auto robotPermissions = context.GetServer()->GetDriveAPI()->GetUserPermissions(context.GetRobotUserId(), TUserPermissionsFeatures());
    if (!robotPermissions) {
        context.AddError("TEvolveSessionAction", "Cannot restore robot permissions for " + context.GetRobotUserId());
        return false;
    }

    const TSessionsFetcher* sessionsFetcher = dynamic_cast<const TSessionsFetcher*>(it->second.Get());
    if (!sessionsFetcher) {
        context.AddError("TEvolveSessionAction", "Cannot get sessions fetcher");
        return false;
    }
    const auto& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags();

    auto sessions = sessionsFetcher->GetSessions();
    for (auto&& sessionData : sessions) {
        const TString& userId = sessionData->GetUserId();
        if (objectIds.contains(userId) && !context.IsFiltered(userId)) {
            TMaybe<TCarTagHistoryEvent> lastEvent = sessionData->GetLastEvent();
            if (!lastEvent) {
                continue;
            }
            if ((*lastEvent)->GetName() != StateFrom) {
                continue;
            }
            auto session = tagsManager.BuildSession(false, true);
            auto eTag = tagsManager.RestoreTag(lastEvent->GetTagId(), session);
            if (!eTag || !*eTag) {
                continue;
            }
            {
                auto newTag = MakeHolder<TChargableTag>(StateTo);
                newTag->SetIsSwitching(true);
                if (!newTag->CopyOnEvolve(**eTag, nullptr, *context.GetServer())) {
                    session.SetErrorInfo("EvolveSessionAction::DoFinish", "cannot CopyOnEvolve");
                    context.AddError("TEvolveSessionAction", "cannot CopyOnEvolve");
                    return false;
                }
                NDrive::ITag::TEvolutionContext forceEvolveContext;
                forceEvolveContext.SetMode(EEvolutionMode::Force);
                if (!tagsManager.EvolveTag(*eTag, std::move(newTag), *robotPermissions, context.GetServer(), session, &forceEvolveContext)) {
                    context.AddError("TEvolveSessionAction", "Cannot evolve tag " + session.GetStringReport());
                    return false;
                }
                if (session.Commit()) {
                    return true;
                }
                context.AddError("TEvolveSessionAction", "Cannot commit evolve tag session " + session.GetStringReport());
                return false;
            }
        }
    }
    return true;
}

bool TEvolveSessionAction::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING(json, "state_from", StateFrom);
    JREAD_STRING(json, "state_to", StateTo);
    return true;
}

NJson::TJsonValue TEvolveSessionAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE(result, "state_from", StateFrom);
    JWRITE(result, "state_to", StateTo);
    return result;
}

NDrive::TScheme TEvolveSessionAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    TVector<TString> tags = { "old_state_reservation", "old_state_acceptance", "old_state_riding", "old_state_parking" };
    scheme.Add<TFSVariants>("state_from", "state_from").SetVariants(tags);
    scheme.Add<TFSVariants>("state_to", "state_to").SetVariants(tags);
    return scheme;
}

TString TEvolveSessionAction::GetActionType() const {
    return "evolve_session";
}
