#include "tag.h"

#include <drive/backend/data/alerts/tags.h>

#include <util/generic/guid.h>


bool TTagActionImpl::HandleObject(const TString objectId, TMap<TString, TVector<TDBTag>>& tagsByObject, NAlerts::TFetcherContext& context) {
    auto patchedMain = INotifierActionBase::PatchData(objectId, AlertTagData, context);
    TMessagesCollector errors;
    const IDriveTagsManager& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager();
    ITag::TPtr patchedAlertTag = IJsonSerializableTag::BuildFromString(tagsManager, patchedMain, &errors);

    if (!ModificationsPtr) {
        context.AddError("TagAction", "Modification nullptr");
        return false;
    }

    switch (TagAction) {
    case EObjectHistoryAction::Add:
        if (!patchedAlertTag) {
            context.AddError("TagAction", "Cant build tag " + patchedMain + " " + errors.GetStringReport());
            return false;
        }
        ModificationsPtr->AddTag(objectId, patchedAlertTag); // EUniquePolicy::SkipIfExists
        break;
    case EObjectHistoryAction::Remove: {
        auto it = tagsByObject.find(objectId);
        if (it != tagsByObject.end()) {
            for (auto&& tag : it->second) {
                ModificationsPtr->RemoveTag(objectId, std::move(tag));
            }
        }
        break;
    }
    case EObjectHistoryAction::UpdateData: {
        if (!patchedAlertTag) {
            context.AddError("TagAction", "Cant build tag " + patchedMain + " " + errors.GetStringReport());
            return false;
        }
        auto it = tagsByObject.find(objectId);
        if (it != tagsByObject.end()) {
            for (auto&& tag : it->second) {
                tag.SetData(patchedAlertTag);
                ModificationsPtr->UpdateTag(objectId, std::move(tag));
            }
        }
        break;
    }
    case EObjectHistoryAction::TagEvolve: {
        if (!patchedAlertTag) {
            context.AddError("TagAction", "Cant build tag " + patchedMain + " " + errors.GetStringReport());
            return false;
        }
        auto it = tagsByObject.find(objectId);
        if (it != tagsByObject.end()) {
            for (auto&& tag : it->second) {
                ModificationsPtr->EvolveTag(objectId, std::move(tag), patchedAlertTag);
            }
        }
        break;
    }
    default:
        context.AddError("TagAction", "Incorrect action type");
        return false;
    }

    for (auto tagJson : AdditionalTags) {
        auto patched = INotifierActionBase::PatchData(objectId, tagJson, context);
        TMessagesCollector errors;
        ITag::TPtr tag = IJsonSerializableTag::BuildFromString(tagsManager, patched, &errors);
        if (!tag) {
            context.AddError("TagAction", "Cant build tag " + patched + " " + errors.GetStringReport());
            return false;
        }
        ModificationsPtr->AddTag(objectId, tag); // EUniquePolicy::Undefined
    }
    return true;
}

TMaybe<NEntityTagsManager::EEntityType> TTagActionImpl::GetTagEntityType() const {
    return NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
}

bool TTagActionImpl::InitAction(const NAlerts::TFetcherContext& context) {
    ModificationsPtr = MakeHolder<TModifications>(context);
    ModificationsPtr->SetFilter(Filter);
    return true;
}

bool TTagActionImpl::CommitAction(NAlerts::TFetcherContext& context) {
    auto tagEntity = GetTagEntityType();
    if (!tagEntity.Defined()) {
        context.AddError("CommitTagAction", "EntityType not defined");
        return false;
    }
    const IEntityTagsManager& entityTagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
    return ModificationsPtr && ModificationsPtr->ApplyModification(entityTagsManager);
}

bool TTagActionImpl::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    auto tagEntity = GetTagEntityType();
    if (!tagEntity.Defined()) {
        context.AddError("CommitTagAction", "EntityType not defined");
        return false;
    }
    const IEntityTagsManager& entityTagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
    if (!InitAction(context)) {
        return false;
    }

    const IDriveTagsManager& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager();
    TString controlTagName = TagName;

    if (!controlTagName && !objectIds.empty()) {
        TMessagesCollector errors;
        ITag::TPtr alertTag = IJsonSerializableTag::BuildFromString(tagsManager, INotifierActionBase::PatchData(*objectIds.begin(), AlertTagData, context), &errors);
        if (!alertTag) {
            context.AddError("TagAction", "Cant build tag " + errors.GetStringReport());
            return false;
        }
        controlTagName = alertTag->GetName();
    }

    if (!controlTagName) {
        context.AddError("TagAction", "TagName not configured");
        return false;
    }

    TVector<TDBTag> taggedObjects;
    {
        auto tx = entityTagsManager.BuildTx<NSQL::ReadOnly>();
        if (!entityTagsManager.RestoreTags({}, { controlTagName }, taggedObjects, tx)) {
            context.AddError("TagAction", "Errors reading tags " + tx.GetStringReport());
            return false;
        }
    }

    TMap<TString, TVector<TDBTag>> commonTagsByObject;
    for (auto&& dbTag : taggedObjects) {
        if (!objectIds.contains(dbTag.GetObjectId())) {
            if (AutoRemove) {
                if (dbTag->GetName() == controlTagName) {
                    if (!ModificationsPtr) {
                        context.AddError("TagAction", "Fail to remove tag without modification ptr");
                        return false;
                    }
                    ModificationsPtr->RemoveTag(dbTag.GetObjectId(), dbTag);
                }
            }
        } else {
            if (TagAction == EObjectHistoryAction::Add) {
                context.SkipObject(dbTag.GetObjectId());
            } else {
                if (dbTag->GetName() == controlTagName) {
                    commonTagsByObject[dbTag.GetObjectId()].emplace_back(std::move(dbTag));
                }
            }
        }
    }

    ui32 updatesCount = 0;
    for (auto&& object : objectIds) {
        if (updatesCount == MaxTags) {
            context.SkipObject(object);
            continue;
        }
        if (context.IsFiltered(object)) {
            continue;
        }
        if (!HandleObject(object, commonTagsByObject, context)) {
            return false;
        }
        ++updatesCount;
    }
    return CommitAction(context);
}

bool TTagActionImpl::DeserializeFromJson(const NJson::TJsonValue& json) {
    AlertTagData = json["tag_data"].GetStringRobust();
    JREAD_STRING_OPT(json, "tag_name", TagName);
    JREAD_BOOL_OPT(json, "auto_remove", AutoRemove);
    JREAD_INT_OPT(json, "max_tags", MaxTags);
    for (auto&& data : json["additional_tags"].GetArray()) {
        if (!data.IsMap() && !data.IsString()) {
            return false;
        }
        AdditionalTags.push_back(data.GetStringRobust());
    }
    JREAD_FROM_STRING_OPT(json, "tag_action", TagAction);
    if (!NJson::ParseField(json["filter"], Filter)) {
        return false;
    }

    return INotifierActionBase::DeserializeFromJson(json);
}

NJson::TJsonValue TTagActionImpl::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE_DEF(result, "auto_remove", AutoRemove, true);
    JWRITE_DEF(result, "tag_name", TagName, "");
    JWRITE(result, "max_tags", MaxTags);
    result["tag_data"] = AlertTagData;
    for (auto&& tag : AdditionalTags) {
        result["additional_tags"].AppendValue(tag);
    }
    JWRITE(result, "tag_action", ::ToString(TagAction));
    result["filter"] = Filter;
    return result;
}

NDrive::TScheme TTagActionImpl::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSString>("tag_data", "Тег для навешивания");
    scheme.Add<TFSString>("tag_name", "Тег для контроля и удаления");
    scheme.Add<TFSString>("filter", "Фильтр тега");
    scheme.Add<TFSBoolean>("auto_remove", "Включить автоудаление тега");
    scheme.Add<TFSNumeric>("max_tags", "Максимальное число тегов за одну итерацию").SetDefault(500);
    scheme.Add<TFSArray>("additional_tags", "Дополнительные теги (навешиваются при срабатывании)").SetElement<TFSString>();
    scheme.Add<TFSVariants>("tag_action", "Действие над тегом").InitVariants<EObjectHistoryAction>();
    return scheme;
}

TMaybe<NEntityTagsManager::EEntityType> TTraceTagAction::GetTagEntityType() const {
    return NEntityTagsManager::EEntityType::Trace;
}

bool TTraceTagAction::DeserializeFromJson(const NJson::TJsonValue& json) {
    for (auto&& data : json["additional_user_tags"].GetArray()) {
        if (!data.IsMap() && !data.IsString()) {
            return false;
        }
        AdditionalUserTags.push_back(data.GetStringRobust());
    }
    return TTagActionImpl::DeserializeFromJson(json);
}

NJson::TJsonValue TTraceTagAction::SerializeToJson() const {
    NJson::TJsonValue result = TTagActionImpl::SerializeToJson();
    for (auto&& tag : AdditionalUserTags) {
        result["additional_user_tags"].AppendValue(tag);
    }
    return result;
}

NDrive::TScheme TTraceTagAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = TTagActionImpl::GetScheme(server);
    scheme.Add<TFSArray>("additional_user_tags", "Дополнительные пользовательские теги (навешиваются при срабатывании)").SetElement<TFSString>();
    return scheme;
}

bool TTraceTagAction::HandleObject(const TString objectId, TMap<TString, TVector<TDBTag>>& tagsByObject, NAlerts::TFetcherContext& context) {
    if (!TTagActionImpl::HandleObject(objectId, tagsByObject, context)) {
        return false;
    }
    const auto tagsPtr = tagsByObject.FindPtr(objectId);
    if (!tagsByObject || tagsPtr->empty()) {
        return true;
    }
    const auto traceTag = tagsPtr->begin()->GetTagAs<TTraceTag>();
    if (!traceTag) {
        context.AddError("TagAction", "Wrong tag type, no trace");
        return false;
    }
    if (GetUuid(traceTag->GetUserId()).IsEmpty()) {
        context.SkipObject(objectId);
        WARNING_LOG << "Wrong tag(" << tagsPtr->begin()->GetTagId() << ") with user id(" << traceTag->GetUserId() << ")" << Endl;
        return true;
    }
    TMap<TString, TString> params;
    {
        auto json = traceTag->SerializeToJson();
        for (auto&& [key, value] : json.GetMap()) {
            params[key] = value.GetStringRobust();
        }
    }
    const IDriveTagsManager& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager();
    for (auto tagJson : AdditionalUserTags) {
        auto patched = INotifierActionBase::PatchData(tagJson, params);
        TMessagesCollector errors;
        ITag::TPtr tag = IJsonSerializableTag::BuildFromString(tagsManager, patched, &errors);
        if (!tag) {
            context.AddError("TagAction", "Cant build tag " + patched + " " + errors.GetStringReport());
            return false;
        }
        UserTagModificationsPtr->AddTag(traceTag->GetUserId(), tag);
    }
    return true;
}

bool TTraceTagAction::InitAction(const NAlerts::TFetcherContext& context) {
    if (!TTagActionImpl::InitAction(context)) {
        return false;
    }
    UserTagModificationsPtr = MakeHolder<TModifications>(context);
    return true;
}

bool TTraceTagAction::CommitAction(NAlerts::TFetcherContext& context) {
    return UserTagModificationsPtr && UserTagModificationsPtr->ApplyModification(context.GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags())
        && TTagActionImpl::CommitAction(context);
}
