#include "telegram.h"

#include <drive/backend/cars/car.h>

INotifierActionBase::TFactory::TRegistrator<TUserNotifierAction> TUserNotifierAction::Registrator("user_notify");
INotifierActionBase::TFactory::TRegistrator<TCarNotifierAction> TCarNotifierAction::Registrator("car_notify");

TString TCarNotifierAction::BuildObjectReport(const TString& objectId, const NAlerts::TFetcherContext& context) const {
    TStringStream ss;
    auto carData = context.GetServer()->GetDriveAPI()->GetCarsData()->GetObject(objectId);
    if (carData) {
        ss << carData->GetHRReport();
    }

    if (GetReportType() != "simple") {
        ss << " (";
        TVector<TTaggedDevice> tags;
        if (context.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetObjectsFromCacheByIds({ objectId }, tags, TInstant::Zero())) {
            for (auto&& obj : tags) {
                for (auto&& tag : obj.GetTags()) {
                    if (tag->GetTagPriority(0) > 0) {
                        ss << tag->GetName() << ",";
                    }
                }
            }
        }
        ss << ")";
    }
    return ss.Str();
}

TString TUserNotifierAction::BuildObjectReport(const TString& objectId, const NAlerts::TFetcherContext& context) const {
    TStringStream ss;
    auto userData = context.GetServer()->GetDriveAPI()->GetUsersData()->GetCachedObject(objectId);
    if (userData) {
        ss << userData->GetHRReport();
    }

    if (GetReportType() != "simple") {
        const auto& userTagManager = context.GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags();
        auto session = userTagManager.BuildSession(true);
        auto optionalTags = userTagManager.RestoreObject(objectId, session);
        ss << " (";
        TVector<TTaggedUser> tags;
        if (optionalTags) {
            {
                for (auto&& tag : optionalTags->GetTags()) {
                    if (tag->GetTagPriority(0) > 0) {
                        ss << tag->GetName() << ",";
                    }
                }
            }
        } else {
            ERROR_LOG << "cannot RestoreObject " << objectId << ": " << session.GetStringReport() << Endl;
        }
        ss << ")";
    }
    return ss.Str();
}

TString TUserNotifierAction::GetActionType() const {
    return "user_notify";
}

TString TCarNotifierAction::GetActionType() const {
    return "car_notify";
}

bool TTelegramAction::DoFinish(const TSet<TString>& objectIds, NAlerts::TFetcherContext& context) {
    auto notifier = context.GetServer()->GetNotifier(GetNotifierName());
    if (!notifier) {
        return false;
    }
    TVector<TString> messages;
    for (auto&& objectId : objectIds) {
        if (!context.IsFiltered(objectId)) {
            if (GetMessageTemplate()) {
                messages.push_back(PatchData(objectId, GetMessageTemplate(), context) + "/" + BuildObjectReport(objectId, context));
            } else {
                messages.push_back(BuildObjectReport(objectId, context));
            }
        }
    }
    if (MultiLine) {
        NDrive::INotifier::MultiLinesNotify(notifier, GetHRCheckerName(), messages);
    } else {
        for (auto&& message : messages) {
            NDrive::INotifier::Notify(notifier, GetHRCheckerName() + ":" + message, NDrive::INotifier::TContext().SetServer(context.GetServer()));
        }
    }
    return true;
}

bool TTelegramAction::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    JREAD_STRING_OPT(jsonInfo, "hr_name", HRCheckerName);
    JREAD_STRING_OPT(jsonInfo, "notifier_name", NotifierName);
    JREAD_STRING_OPT(jsonInfo, "message_template", MessageTemplate);
    JREAD_STRING_OPT(jsonInfo, "report_type", ReportType);
    JREAD_BOOL_OPT(jsonInfo, "multi_line", MultiLine);
    return INotifierActionBase::DeserializeFromJson(jsonInfo);
}

NJson::TJsonValue TTelegramAction::SerializeToJson() const {
    NJson::TJsonValue result = INotifierActionBase::SerializeToJson();
    JWRITE_DEF(result, "hr_name", HRCheckerName, "");
    JWRITE_DEF(result, "notifier_name", NotifierName, "");
    JWRITE_DEF(result, "message_template", MessageTemplate, "");
    JWRITE_DEF(result, "report_type", ReportType, "simple");
    JWRITE_DEF(result, "multi_line", MultiLine, true);
    return result;
}

NDrive::TScheme TTelegramAction::GetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme = INotifierActionBase::GetScheme(server);
    scheme.Add<TFSVariants>("notifier_name", "Нотификатор").SetVariants(server.GetNotifierNames());
    scheme.Add<TFSString>("message_template", "Шаблон сообщения");
    scheme.Add<TFSBoolean>("multi_line", "Многострочное сообщение");
    scheme.Add<TFSVariants>("report_type", "Вид репорта об объкте").SetVariants({"simple", "tags"});
    scheme.Add<TFSString>("hr_name", "Заголовок сообщения в чате");
    return scheme;
}
