#include "condition.h"

using namespace NAlerts;

bool TCheckersSet::ApplyFilters(const IServiceDataFetcherConfig::TItems& dataFetcherConfigs,
                                                                      TFetcherContext& context,
                                                                      ICheckerAction& func) const
{
    TVector<IServiceDataFetcher::TPtr> fetchers(dataFetcherConfigs.size());
    for (const auto& [_, configPtr] : dataFetcherConfigs) {
        IServiceDataFetcher::TPtr fetcher = configPtr->BuildFetcher();
        if (!fetcher) {
            context.AddError("ApplyFilters", "Cant build fetcher");
            return false;
        }
        fetchers[configPtr->GetIndex()] = fetcher;
        context.AddFetcher(configPtr->GetFetcherType(), fetcher);
    }

    for (size_t i = 0; i < fetchers.size(); ++i) {
        auto fetcher = fetchers[i];
        if (!fetcher->Fetch(context)) {
            return false;
        }
        auto iterators = fetcher->BuildIterators(context);
        if (!iterators) {
            return false;
        }

        bool checkResult = true;
        if (i < fetchers.size() - 1) {
            checkResult = CheckFetched(*iterators, context, func, fetcher->GetEntityType(), fetchers[i+1]);
        } else {
            func.SetEntityType(fetcher->GetEntityType());
            checkResult = CheckFetched(*iterators, context, func, fetcher->GetEntityType(), nullptr);
        }
        if (!checkResult) {
            return false;
        }
    }
    return true;
}

bool TCheckersSet::CheckFetched(const TVector<IFetchedIterator::TPtr>& iterators, TFetcherContext& context, ICheckerAction& func, const EAlertEntityType entityType, IServiceDataFetcher::TPtr nextFetcher) const {
    if (iterators.empty()) {
        context.AddError("CheckFetched", "No iterators for fetcher");
        return false;
    }
    TSet<TStringBuf> objectIds;
    for (auto&& items : iterators) {
        if (!items->IsFinished()) {
            objectIds.insert(items->GetObjectId());
        }
    }

    if (objectIds.empty() && nextFetcher) {
        context.AddToLog(iterators[0]->GetFetcherName() + " object set is empty");
        nextFetcher->SetEmptyFilters();
    }

    while (!objectIds.empty()) {
        if (objectIds.size() != 1) {
            context.AddError("iterators", TStringBuilder() << "common_alerts inconsistent objects " << context.GetCurrentState());
            return false;
        }
        TStringBuf currentObject = *(objectIds.begin());
        TFetchedMap objectItems;

        bool correctValue = true;
        for (ui32 i = 0; i < iterators.size(); ++i) {
            auto& it = iterators[i];
            if (it->IsFinished()) {
                context.AddError("iterators", TStringBuilder() << "common_alerts inconsistent iterator " << context.GetCurrentState() << " iterator " << it->GetFetcherName());
                return false;
            }
            if (it->GetObjectId() != currentObject) {
                context.AddError("iterators", TStringBuilder() << "common_alerts wrong data " << context.GetCurrentState() << " iterator " << it->GetFetcherName() << " current object " << currentObject << ", iterator " << it->GetObjectId());
                return false;
            }
            if (correctValue) {
                auto value = it->Get();
                if (!value.Defined()) {
                    context.AddToLog(it->GetFetcherName() + " could not fetch value for object " + currentObject + " errors: " + context.GetErrors().GetStringReport());
                }
                auto fetchedValue = value.Defined() ? *value : IFetchedIterator::InvalidData();
                context.AddToLog(it->GetFetcherName() + " fetched value " + ToString(fetchedValue) + " for object " + currentObject);
                auto checker = it->GetChecker();
                if (!checker || !checker->Check(fetchedValue)) {
                    context.AddToLog(it->GetFetcherName() + " bad value " + ToString(fetchedValue) + " for object " + currentObject);
                    correctValue = false;
                }
                if (it->GetSubstitutionName()) {
                    objectItems.emplace(it->GetSubstitutionName(), fetchedValue);
                }
            }
            if (correctValue && i == iterators.size() - 1 && nextFetcher) {
                nextFetcher->AddFetchedIdFilters(it, entityType);
            }
            while (!it->IsFinished() && it->GetObjectId() == currentObject) {
                it->Next();
            }
            if (!it->IsFinished()) {
                objectIds.insert(it->GetObjectId());
            }
        }
        if (correctValue && !nextFetcher) {
            func.Apply(currentObject, objectItems, context);
        }

        objectIds.erase(currentObject);
    }
    return true;
}
