#include "container.h"

#include <drive/backend/data/common/serializable.h>
#include <drive/backend/database/drive_api.h>

#include <rtline/library/unistat/cache.h>

#include <util/string/join.h>


namespace {
    NDrive::TScheme GetMetaDataScheme(const IServerBase& /*server*/) {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("group_name", "Название группы алертов");
        scheme.Add<TFSStructure>("sharding_policy", "Условия для выбора объекта").SetStructure<NDrive::TScheme>(TObjectSharding::GetScheme());
        return scheme;
    }
    const NUnistat::TIntervals IntervalsRobotTimes = { 0, 1, 2, 3, 4, 5, 10, 15, 20, 25, 30,
        40, 50, 60, 70, 80, 90, 100, 125, 150, 175, 200, 225, 250, 300, 350, 400,
        500, 600, 700, 800, 900, 1000, 1500, 2000, 3000, 5000, 10000, 11000, 12000, 13000, 14000, 15000, 20000 };
}

void NAlerts::TAlertContainer::GetScheme(const IServerBase& /*server*/, NDrive::TScheme& scheme) {
    scheme.Add<TFSJson>("action", "Действие").SetDefault(R"({
        "actions": [],
        "action_type": "composite"
    })");
    scheme.Add<TFSString>("group_name", "Название группы алертов");
    scheme.Add<TFSStructure>("sharding_policy", "Условия для выбора объекта").SetStructure<NDrive::TScheme>(TObjectSharding::GetScheme());
    auto checkerScheme = IDataIntervalChecker::GetDefaultScheme();
    checkerScheme.Add<TFSJson>("meta", "Конфигурация");

    checkerScheme.Add<TFSVariants>("fetcher_type", "Фетчер, к которому относится итератор").SetVariants(GetEnumAllValues<EDataFetcherType>());
    checkerScheme.Add<TFSVariants>("iterator_type", "Тип итератора").SetVariants(GetEnumAllValues<NAlerts::EFetchedItems>());
    scheme.Add<TFSArray>("checkers", "Чекеры").SetElement(checkerScheme);

    NDrive::TScheme fetcherScheme;
    fetcherScheme.Add<TFSJson>("config", "Конфигурация");
    scheme.Add<TFSArray>("fetchers", "Фетчеры").SetElement(fetcherScheme);
    scheme.Add<TFSBoolean>("debug_log", "Отладочное логирование действий");
}

bool NAlerts::TAlertContainer::DeserializeFromJson(const NJson::TJsonValue& jsonInfo) {
    NJson::TJsonValue filtersJson = NJson::JSON_ARRAY;
    NJson::TJsonValue itemsJson = NJson::JSON_ARRAY;
    NJson::TJsonValue metaJson;

    if (!NJson::ParseField(jsonInfo, "debug_log", DebugLogEnabled, false)) {
        return false;
    }
    if (jsonInfo.Has("action")) {
        ActionConfig = jsonInfo["action"];
    }
    const auto& fetchers = jsonInfo["fetchers"];
    if (fetchers.IsDefined()) {
        filtersJson = fetchers;
    }
    const auto& checkers = jsonInfo["checkers"];
    if (checkers.IsDefined()) {
        itemsJson = checkers;
    }
    metaJson = jsonInfo;

    if (!ActionConfig.IsMap() || !filtersJson.IsArray() || !itemsJson.IsArray() || !metaJson.IsMap()) {
        return false;
    }

    JREAD_STRING_OPT(metaJson, "group_name", AlertGroupName);
    if (metaJson.Has("sharding_policy") && !ShardingPolicy.DeserializeFromJson(metaJson["sharding_policy"])) {
        return false;
    }

    auto action = INotifierActionBase::BuildFromJson(ActionConfig);
    if (!action) {
        return false;
    }
    i32 fetcherIndex = 0;
    TVector<IServiceDataFetcherConfig::TPtr> configs(filtersJson.GetArray().size(), nullptr);
    for (auto&& fetcherJson : filtersJson.GetArray()) {
        auto fetcherConfig = IServiceDataFetcherConfig::BuildFromJson(fetcherJson["config"]);
        if (!fetcherConfig) {
            return false;
        }
        if (fetcherConfig->GetIndex() <= 0) {
            fetcherConfig->SetIndex(fetcherIndex++);
        }
        if (fetcherConfig->GetIndex() >= (i32) configs.size() || configs[fetcherConfig->GetIndex()]) {
            return false;
        }
        configs[fetcherConfig->GetIndex()] = fetcherConfig;
    }

    for (ui32 i = 0; i < configs.size(); ++i) {
        if (!configs[i]) {
            return false;
        }
        if (i != 0) {
            if (!configs[i]->GetAcceptableEntityTypes().contains(configs[i-1]->GetEntityType())) {
                return false;
            }
        }
        if (!DataFetchers.insert({ configs[i]->GetFetcherType(), configs[i] }).second) {
            return false;
        }
    }
    TMap<EDataFetcherType, ui32> iteratorsForFetcher;
    for (auto&& conditionJson : itemsJson.GetArray()) {
        TString fetcherFull, iteratorName, fetcherName;
        EDataFetcherType fetcherType;
        NAlerts::EFetchedItems iteratorType;
        if (conditionJson.Has("fetcher_type") && conditionJson.Has("iterator_type")) {
            if (!NJson::ParseField(conditionJson, "fetcher_type", NJson::Stringify(fetcherType), true) || !NJson::ParseField(conditionJson, "iterator_type", NJson::Stringify(iteratorType), true)) {
                return false;
            }
            fetcherName = ToString(fetcherType);
            iteratorName = ToString(iteratorType);
        } else if (NJson::ParseField(conditionJson, "fetcher", fetcherFull, true)) {
            TVector<TString> parts = SplitString(fetcherFull, ".");
            if (parts.size() != 2
                || !TryFromString(parts[0], fetcherType)
                || !TryFromString(parts[1], iteratorType)) {
                return false;
            }
            fetcherName = parts[0];
            iteratorName = parts[1];
        } else {
            return false;
        }

        auto it = DataFetchers.find(fetcherType);
        if (it == DataFetchers.end()) {
            return false;
        }

        IDataIntervalChecker::TPtr condition = IDataIntervalChecker::Construct(iteratorName);
        if (!condition || !condition->DeserializeFromJson(conditionJson)) {
            return false;
        }

        IIteratorConfig::TPtr configInstance = IIteratorConfig::TFactory::Construct(iteratorType, iteratorType, condition);
        if (!configInstance) {
            return false;
        }
        if (conditionJson.Has("meta") && !configInstance->DeserializeFromJson(conditionJson["meta"])) {
            return false;
        }
        it->second->AddIteratorConfig(configInstance);
        ++iteratorsForFetcher[fetcherType];
    }
    if (iteratorsForFetcher.size() != DataFetchers.size()) {
        return false;
    }
    return true;
}

void NAlerts::TAlertContainer::SerializeToJson(NJson::TJsonValue& result) const {
    NJson::TJsonValue filtersJson = NJson::JSON_ARRAY;
    NJson::TJsonValue itemsJson = NJson::JSON_ARRAY;

    for (auto&& [fetcherType, fetcher] : DataFetchers) {
        NJson::TJsonValue configJson;
        configJson["config"] = fetcher->SerializeToJson();
        filtersJson.AppendValue(configJson);

        for (auto&& iteratorConfig : fetcher->GetIteratorConfigs()) {
            NJson::TJsonValue iteratorJson = NJson::JSON_MAP;
            if (iteratorConfig->GetChecker()) {
                iteratorJson = iteratorConfig->GetChecker()->SerializeToJson();
            }
            iteratorJson["fetcher_type"] = ToString(fetcher->GetFetcherType());
            iteratorJson["iterator_type"] = ToString(iteratorConfig->GetIteratorType());
            iteratorJson["fetcher"] = ToString(fetcher->GetFetcherType()) + "." + ToString(iteratorConfig->GetIteratorType());
            iteratorJson["meta"] = iteratorConfig->SerializeToJson();
            itemsJson.AppendValue(iteratorJson);
        }
    }

    NJson::InsertField(result, "group_name", AlertGroupName);
    NJson::InsertField(result, "sharding_policy", ShardingPolicy.SerializeToJson());
    NJson::InsertField(result, "debug_log", DebugLogEnabled);
    result["action"] = ActionConfig;
    result["fetchers"] = filtersJson;
    result["checkers"] = itemsJson;
}

bool NAlerts::TAlertContainer::Execute(const NDrive::IServer* server, const TInstant startInstant, ICheckerAction* actionExt, const TString& stateName, TAtomicSharedPtr<IRTBackgroundProcessState> state) const {
    TFetcherContext fetcherContext(server, EntityType, DebugLogEnabled);
    fetcherContext.SetFetchInstant(startInstant);
    fetcherContext.SetRobotUserId(RobotUserId);
    fetcherContext.SetCurrentState(stateName);
    fetcherContext.SetShardingPolicy(ShardingPolicy);
    fetcherContext.SetRobotState(state);
    INotifierActionBase::TPtr action = INotifierActionBase::BuildFromJson(ActionConfig);
    Y_ENSURE_BT(action);
    ICheckerAction& callback = actionExt ? *actionExt : *action;

    TInstant tsBreak = Now();
    if (!Checker.ApplyFilters(DataFetchers, fetcherContext, callback)) {
        Errors.AddMessage<TLOG_ERR>("container", "Can't fetch data: "  + fetcherContext.GetErrors().GetStringReport());
        Errors.MergeMessages(fetcherContext.GetErrors(), "runtime");
        TUnistatSignalsCache::SignalAdd("universal-robot-prepare-" + stateName, "errors", 1);
        return false;
    }
    bool finishResult = callback.Finish(fetcherContext);
    TUnistatSignalsCache::SignalHistogram("universal-robot-finish-" + stateName, "times", (Now() - tsBreak).Seconds(), IntervalsRobotTimes);
    if (!finishResult) {
        Errors.AddMessage<TLOG_ERR>("container", "Action runtime error: " + fetcherContext.GetErrors().GetStringReport());
        Errors.MergeMessages(fetcherContext.GetErrors(), "runtime");
        TUnistatSignalsCache::SignalAdd("universal-robot-finish-" + stateName, "errors", 1);
        return false;
    }
    TUnistatSignalsCache::SignalAdd("universal-robot-finish-" + stateName, "ok", 1);
    return true;
}

NJson::TJsonValue NAlerts::TAlertContainer::GetMetaInfo(const IServerBase& server) {
    NJson::TJsonValue result;
    {
        NJson::TJsonValue actionsJson;
        TSet<TString> registeredActions;
        INotifierActionBase::TFactory::GetRegisteredKeys(registeredActions);
        for (auto&& actionType : registeredActions) {
            THolder<INotifierActionBase> actionImpl = THolder(INotifierActionBase::TFactory::Construct(actionType));
            Y_ENSURE_BT(actionImpl);
            auto scheme = actionImpl->GetScheme(server);
            NJson::TJsonValue& metaBlock = actionsJson[actionType];
            metaBlock["scheme"] = scheme.SerializeToJson();
            metaBlock["title"] = "Действия";
            metaBlock["description"] = "Здесь будет крутое описание ";
        }
        result["actions"] = actionsJson;
    }
    {
        NJson::TJsonValue filtersJson;
        TSet<EDataFetcherType> registeredFilters;
        IServiceDataFetcherConfig::TFactory::GetRegisteredKeys(registeredFilters);
        for (auto&& filterType : registeredFilters) {
            THolder<IServiceDataFetcherConfig> filterImpl = THolder(IServiceDataFetcherConfig::TFactory::Construct(filterType));
            Y_ENSURE_BT(filterImpl);
            auto scheme = filterImpl->GetScheme(server);
            NJson::TJsonValue& metaBlock = filtersJson[::ToString(filterType)];
            metaBlock["scheme"] = scheme.SerializeToJson();
            metaBlock["title"] = "Первичные фильтры";
            metaBlock["description"] = "Здесь будет крутое описание ";

            for (auto&& item : GetEnumNames<NAlerts::EFetchedItems>()) {
                metaBlock["acceptable_items"].AppendValue(item.second);
            }

        }
        result["filters"] = filtersJson;
    }
    {
        NJson::TJsonValue itemsJson;
        for (auto&& item : GetEnumNames<NAlerts::EFetchedItems>()) {
            IDataIntervalChecker::TPtr interval =  IDataIntervalChecker::Construct(item.second);
            if (!interval) {
                continue;
            }
            auto scheme = interval->GetScheme();
            IIteratorConfig::TPtr configInstance = IIteratorConfig::TFactory::Construct(item.first, item.first, interval);
            if (!!configInstance) {
                scheme.Add<TFSStructure>("meta", "Конфигурация").SetStructure(configInstance->GetScheme(server));
            }

            NJson::TJsonValue& metaBlock = itemsJson[item.second];
            metaBlock["scheme"] = scheme.SerializeToJson();
            metaBlock["title"] = "Вторичные фильтры (диапазоны)";
            metaBlock["description"] = "Здесь будет самое крутое описание ";
        }
        result["items"] = itemsJson;
    }
    {
        NJson::TJsonValue metaBlock;
        metaBlock["scheme"] = GetMetaDataScheme(server).SerializeToJson();
        metaBlock["title"] = "Метаданные алерта";
        metaBlock["description"] = "Здесь будет самое крутое описание ";
        result["meta_info"] = metaBlock;
    }
    return result;
}

template <class T>
TMaybe<T> NAlerts::TAlertContainer::CheckAlertSetting(const TString& settingName, const TString& alertName, const NDrive::IServer* server) const {
    auto concreteSetting = server->GetSettings().GetValue<T>("common_alerts." + settingName + "." + alertName);
    if (concreteSetting.Defined()) {
        return concreteSetting;
    }
    return server->GetSettings().GetValue<T>("common_alerts." + settingName);
}
