#include "billing.h"

#include <drive/backend/data/billing_tags.h>


namespace NAlerts {
    IFetchedIterator::TFactory::TRegistrator<TBillingSumIterator> TBillingSumIterator::Registrator(EFetchedItems::BillingTagSum);
    IFetchedIterator::TFactory::TRegistrator<TDebtAgeIterator> TDebtAgeIterator::Registrator(EFetchedItems::BillingDebtAge);
    IFetchedIterator::TFactory::TRegistrator<TCreditCardsCountIterator> TCreditCardsCountIterator::Registrator(EFetchedItems::CreditCardsCount);

    bool TDebtIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return TJsonProcessor::ReadContainer(json, "billing_type", BillingTypes);
    }

    NJson::TJsonValue TDebtIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        TJsonProcessor::WriteContainerArrayStrings(result, "billing_type", BillingTypes);;
        return result;
    }

    NDrive::TScheme TDebtIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSVariants>("billing_type", "Тип списаний (терминал)").InitVariants<EBillingType>().SetMultiSelect(true);
        return scheme;
    }

    TMaybe<TBillingTasks> InitDebtTasks(IFetchedIterator& objectIterator, const TBillingManager& billingManager, const TSet<EBillingType>& billingTypes, NDrive::TEntitySession& session) {
        TSet<TString> users;
        while (!objectIterator.IsFinished()) {
            auto object = objectIterator.GetObjectId(EAlertEntityType::User);
            users.emplace(object.Data(), object.Size());
            objectIterator.Next();
        }
        if (users.empty()) {
            return TBillingTasks();
        }
        NSQL::TQueryOptions queryOptions;
        queryOptions.SetGenericCondition("user_id", std::move(users));
        if (billingTypes) {
            TSet<TString> types;
            for(auto&& type : billingTypes) {
                types.emplace(ToString(type));
            }
            queryOptions.SetGenericCondition("billing_type", std::move(types));
        }
        queryOptions.AddGenericCondition("task_status", ::ToString(EPaymentStatus::NoFunds));
        queryOptions.AddGenericCondition("task_status", ::ToString(EPaymentStatus::NoCards));

        return billingManager.GetActiveTasksManager().GetTasks(session, queryOptions.PrintConditions(*session.GetTransaction()));
    }

    bool TBillingSumIterator::InitByObjects(IFetchedIterator& objectIterator) {
        auto config = TBase::template GetConfigAs<TDebtIteratorConfig>();
        if (!config) {
            return false;
        }
        if (!TBase::Context.GetServer()->GetDriveAPI()->HasBillingManager()) {
            return false;
        }

        const auto& billingManager = TBase::Context.GetServer()->GetDriveAPI()->GetBillingManager();
        auto session = billingManager.BuildSession(true);
        auto tasks = InitDebtTasks(objectIterator, billingManager, config->GetBillingTypes(), session);
        if (!tasks) {
            TBase::Context.AddError("GetTasks", session.GetStringReport());
            return false;
        }
        if (tasks->empty()) {
            return true;
        }
        auto payments = billingManager.GetPaymentsManager().GetSessionsPayments(*tasks, session, true);
        if (!payments) {
            TBase::Context.AddError("GetSessionsPayments", session.GetStringReport());
            return false;
        }
        for (auto&& [sessionId, payment] : *payments) {
            PaymentsData[payment.GetBillingTask().GetUserId()].emplace_back(std::move(payment));
        }
        return true;
    }

    bool TDebtAgeIterator::InitByObjects(IFetchedIterator& objectIterator) {
        auto config = TBase::template GetConfigAs<TDebtIteratorConfig>();
        if (!config) {
            return false;
        }
        if (!TBase::Context.GetServer()->GetDriveAPI()->HasBillingManager()) {
            return false;
        }
        const auto& billingManager = TBase::Context.GetServer()->GetDriveAPI()->GetBillingManager();
        auto session = billingManager.BuildSession(true);
        auto tasks =  InitDebtTasks(objectIterator, billingManager, config->GetBillingTypes(), session);
        if (!tasks) {
            TBase::Context.AddError("GetTasks", session.GetStringReport());
            return false;
        }
        if (tasks->empty()) {
            return true;
        }
        TSet<TString> sessionIds;
        for (auto&& task : *tasks) {
            sessionIds.emplace(task.GetId());
        }
        NSQL::TQueryOptions paymentsQueryOptions;
        paymentsQueryOptions.SetGenericCondition("session_id", std::move(sessionIds));
        paymentsQueryOptions.AddCustomCondition("True) GROUP BY (session_id");

        TRecordsSet records;
        auto transaction = session.GetTransaction();
        auto query = paymentsQueryOptions.PrintQuery(*transaction, "drive_payments", {"min(id) as id", "session_id"});
        auto queryResult = transaction->Exec(query, &records);
        if (!queryResult || !queryResult->IsSucceed()) {
            TBase::Context.AddError("GetMinimalPayments", transaction->GetErrors().GetStringReport());
            return false;
        }

        TSet<ui64> paymentIds;
        for (auto && record : records) {
            ui64 paymentId = 0;
            if (!TryFromString(record.Get("id"), paymentId)) {
                TBase::Context.AddError("GetMinimalPayments", "incorrect id " + record.Get("id"));
                return false;
            }
            paymentIds.emplace(paymentId);
        }

        TMap<TString, TCachedPayments> payments;
        if (!billingManager.GetPaymentsManager().GetPayments(payments, NSQL::TQueryOptions().SetGenericCondition("id", paymentIds).PrintConditions(*transaction), session, false)) {
            TBase::Context.AddError("GetPayments", session.GetStringReport());
            return false;
        }

        for (auto&& task : *tasks) {
            auto it = payments.find(task.GetId());
            if (it != payments.end()) {
                Payments[task.GetUserId()].emplace_back(it->second);
            }
        }

        return true;
    }
}
