#pragma once

#include "iterators.h"

#include <drive/backend/billing/manager.h>
#include <drive/backend/billing/trust/charge_logic.h>

namespace NAlerts {

    class TDebtIteratorConfig : public IIteratorConfig {
        using TBase = IIteratorConfig;
        R_READONLY(TSet<EBillingType>, BillingTypes);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TBillingSumIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TBillingSumIterator> Registrator;
        TMap<TString, TVector<TPaymentsData>> PaymentsData;

    public:
        using TBase::TBase;

        virtual bool InitByObjects(IFetchedIterator& objectIterator) override;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::BillingTagSum;
        }

        virtual bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TDebtIteratorConfig>();
            if (!config) {
                return false;
            }
            auto object = TBase::GetObjectId(EAlertEntityType::User);
            TString userId(object.Data(), object.Size());
            auto it = PaymentsData.find(userId);
            if (it == PaymentsData.end()) {
                data = 0;
            } else {
                data = TBase::Context.GetServer()->GetDriveAPI()->GetBillingManager().GetDebt(it->second, nullptr, config->GetBillingTypes());
            }
            return true;
        }
    };

    class TDebtAgeIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TDebtAgeIterator> Registrator;
        TMap<TString, TVector<TCachedPayments>> Payments;
    public:
        using TBase::TBase;
        virtual EFetchedItems GetField() const override {
            return EFetchedItems::BillingDebtAge;
        }
        virtual bool InitByObjects(IFetchedIterator& objectIterator) override;

        virtual bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TDebtIteratorConfig>();
            if (!config) {
                return false;
            }

            TInstant debtInstant = TInstant::Max();
            auto object = TBase::GetObjectId(EAlertEntityType::User);
            TString userId(object.Data(), object.Size());
            auto it = Payments.find(userId);
            if (it == Payments.end()) {
                return true;
            }
            for (auto&& payments : it->second) {
                debtInstant = Min(debtInstant, payments.GetFirstPaymentTs());
            }
            if (TBase::Context.GetFetchInstant() > debtInstant) {
                data = (TBase::Context.GetFetchInstant() - debtInstant).Seconds();
            }
            return true;
        }
    };

    class TCreditCardsCountIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TCreditCardsCountIterator> Registrator;
    public:
        using TBase::TBase;
        using TBase::GetObjectId;
        using TBase::Context;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::CreditCardsCount;
        }

        virtual bool ExtractData(TFetchedValue& data) const override {
            TString userId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
            auto permissions = TBase::Context.GetServer()->GetDriveAPI()->GetUserPermissions(userId, TUserPermissionsFeatures());
            if (!permissions) {
                return false;
            }
            auto methods = TBase::Context.GetServer()->GetDriveAPI()->GetUserPaymentMethodsSync(*permissions, *TBase::Context.GetServer(), false);
            auto userCards = TBillingManager::GetUserPaymentCards(methods.Get());
            if (!userCards.Defined()) {
                return false;
            }
            data = userCards->size();
            return true;
        }
    };
}
