#include "chat_info.h"

namespace NAlerts {

    IFetchedIterator::TFactory::TRegistrator<TNotReadMessagesCount> TNotReadMessagesCount::Registrator(EFetchedItems::ChatUnreadMessages);
    IFetchedIterator::TFactory::TRegistrator<TIsNewChat> TIsNewChat::Registrator(EFetchedItems::ChatIsNew);
    IFetchedIterator::TFactory::TRegistrator<TChatLastMessageDeltaIterator> TChatLastMessageDeltaIterator::Registrator(EFetchedItems::ChatLastMessageDelta);

    bool TChatInfoIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "chat_robot", ChatRobot) &&
            NJson::ParseField(json, "chat_topic", Topic) &&
            NJson::ParseField(json, "depth", Depth);
    }

    NJson::TJsonValue TChatInfoIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "chat_robot", ChatRobot);
        NJson::InsertField(result, "chat_topic", Topic);
        NJson::InsertField(result, "depth", Depth);
        return result;
    }

    NDrive::TScheme TChatInfoIteratorConfig::GetScheme(const IServerBase& server) const {
        NDrive::TScheme scheme;
        auto robots = server.GetAs<NDrive::IServer>()->GetChatRobots();
        if (robots) {
            scheme.Add<TFSVariants>("chat_robot", "Тип робота").SetVariants(NContainer::Keys(robots));
        } else {
            scheme.Add<TFSVariants>("chat_robot", "Тип робота");
        }
        scheme.Add<TFSString>("chat_topic", "Топик");
        scheme.Add<TFSDuration>("depth", "Глубина поиска");
        return scheme;
    }

    TExpected<NDrive::NChat::TChatMessages, TString> GetChatMessages(const TChatInfoIteratorConfig* config) {
        if (!config) {
            return MakeUnexpected<TString>("no config");
        }
        auto chatRobot = NDrive::GetServerAs<NDrive::IServer>().GetChatRobot(config->GetChatRobot());
        if (!chatRobot) {
            return MakeUnexpected<TString>(config->GetChatRobot() + " chat robot not configured");
        }
        auto session = chatRobot->BuildChatEngineSession(true);

        auto chats = chatRobot->GetChats(config->GetChatRobot(), config->GetTopic(), session);
        if (!chats) {
            return MakeUnexpected<TString>("cannot get chats " + session.GetStringReport());
        }
        auto messagesByChats = chatRobot->GetUserChatsMessages(*chats, session, {}, {Now() - config->GetDepth()});
        if (!messagesByChats) {
            return MakeUnexpected<TString>("cannot get chat messages " + session.GetStringReport());
        }
        return *messagesByChats;
    }

    TNotReadMessagesCount::TNotReadMessagesCount(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig)
    : TBase(fetcherType, entityType, context, iteratorConfig)
    {
    }

    bool TNotReadMessagesCount::InitByObjects(NAlerts::IFetchedIterator& /* fetcher */) {
        auto config = TBase::template GetConfigAs<TChatInfoIteratorConfig>();
        Y_ENSURE_BT(config);
        auto messages = GetChatMessages(config);
        if (!messages) {
            ERROR_LOG << "TChatLastMessageDeltaIterator: " << messages.GetError() << Endl;
            return false;
        }

        ChatMessages = std::move(*messages);
        return true;
    }

    bool TNotReadMessagesCount::ExtractData(TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TChatInfoIteratorConfig>();
        if (!config) {
            return false;
        }

        const TFetcherContext& context = TBase::Context;
        auto chatRobot = context.GetServer()->GetChatRobot(config->GetChatRobot());
        if (!chatRobot) {
            context.AddError("TNotReadMessagesCount", config->GetChatRobot() + " chat robot not configured");
            return false;
        }

        TString userId(TBase::GetObjectId().data(), TBase::GetObjectId().size());
        auto searchId = chatRobot->GetChatSearchId(userId, config->GetTopic());
        auto messagesIt = ChatMessages.find(searchId);
        if (messagesIt == ChatMessages.end()) {
            return true;
        }

        auto lastViewedMessage = chatRobot->GetLastViewedMessageId(userId, config->GetTopic(), userId);
        data = 0;
        for (auto&& message : messagesIt->second) {
            if (message.GetHistoryEventId() > lastViewedMessage) {
                ++data;
            }
        }
        return true;
    }

    TChatLastMessageDeltaIterator::TChatLastMessageDeltaIterator(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig)
        : TBase(fetcherType, entityType, context, iteratorConfig)
    {
    }

    bool TChatLastMessageDeltaIterator::InitByObjects(NAlerts::IFetchedIterator& /* fetcher */) {
        auto config = TBase::template GetConfigAs<TChatInfoIteratorConfig>();
        Y_ENSURE_BT(config);

        auto messages = GetChatMessages(config);
        if (!messages) {
            ERROR_LOG << "TChatLastMessageDeltaIterator: " << messages.GetError() << Endl;
            return false;
        }

        ChatMessages = std::move(*messages);
        return true;
    }

    bool TChatLastMessageDeltaIterator::ExtractData(TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TChatInfoIteratorConfig>();
        if (!config) {
            return false;
        }

        const TFetcherContext& context = TBase::Context;
        auto chatRobot = context.GetServer()->GetChatRobot(config->GetChatRobot());
        if (!chatRobot) {
            ERROR_LOG << config->GetChatRobot() << " chat robot not configured" << Endl;
            return false;
        }
        TString userId(TBase::GetObjectId().data(), TBase::GetObjectId().size());
        auto searchId = chatRobot->GetChatSearchId(userId, config->GetTopic());
        auto messagesIt = ChatMessages.find(searchId);
        if (messagesIt == ChatMessages.end() || messagesIt->second.empty()) {
            return true;
        }
        TInstant lastMessageInstant = messagesIt->second.back().GetHistoryInstant();
        data = (context.GetFetchInstant() - lastMessageInstant).Seconds();
        return true;
    }
}
