#pragma once
#include "iterators.h"

#include <drive/backend/chat_robots/abstract.h>

namespace NAlerts {

    class TChatInfoIteratorConfig : public IIteratorConfig {
        using TBase = IIteratorConfig;
        R_READONLY(TString, ChatRobot);
        R_READONLY(TString, Topic);
        R_READONLY(TDuration, Depth, TDuration::Days(30));

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TNotReadMessagesCount : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TNotReadMessagesCount> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::ChatUnreadMessages;
        }

        TNotReadMessagesCount(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig);
        bool ExtractData(TFetchedValue& data) const override;
        bool InitByObjects(NAlerts::IFetchedIterator& fetcher) override;

    private:
        NDrive::NChat::TChatMessages ChatMessages;
    };

    class TIsNewChat : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TIsNewChat> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::ChatIsNew;
        }

        bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TChatInfoIteratorConfig>();
            if (!config) {
                return false;
            }

            const TFetcherContext& context = TBase::Context;
            auto chatRobot = context.GetServer()->GetChatRobot(config->GetChatRobot());
            if (!chatRobot) {
                context.AddError("TIsNewChat", config->GetChatRobot() + " chat robot not configured");
                return false;
            }

            TString userId(TBase::GetObjectId().data(), TBase::GetObjectId().size());
            if (chatRobot->GetLastViewedMessageId(userId, config->GetTopic(), userId) == 0) {
                data = 1;
            } else {
                data = 0;
            }
            return true;
        }
    };

    class TChatLastMessageDeltaIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TChatLastMessageDeltaIterator> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::ChatLastMessageDelta;
        }

        TChatLastMessageDeltaIterator(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig);
        bool ExtractData(TFetchedValue& data) const override;
        bool InitByObjects(NAlerts::IFetchedIterator& fetcher) override;

    private:
        NDrive::NChat::TChatMessages ChatMessages;
    };
}
