#pragma once

#include "iterators.h"

#include <drive/backend/compiled_riding/manager.h>

namespace NAlerts {

    class TCompiledRidesIteratorConfig : public IIteratorConfig {
        using TBase = IIteratorConfig;
        /*
        R_READONLY(TDuration, RidingMinTime, TDuration::Zero());
        R_READONLY(TDuration, RidingMaxTime, TDuration::Max());
        R_READONLY(TDuration, ParkingMinTime, TDuration::Zero());
        R_READONLY(TDuration, ParkingMaxTime, TDuration::Max());
        R_READONLY(TDuration, AcceptanceMinTime, TDuration::Zero());
        R_READONLY(TDuration, AcceptanceMaxTime, TDuration::Max());
        R_READONLY(TDuration, HistoryDeep, TDuration::Days(1));
        */

        R_READONLY(TDuration, StartDeep, TDuration::Days(1));
        R_READONLY(TDuration, FinishDeep, TDuration::Zero());
        // Aggregation (sessions count)

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TCompiledRidesCount : public TContainerIteratorBase {
    private:
        using TBase = TContainerIteratorBase;

    private:
        static IFetchedIterator::TFactory::TRegistrator<TCompiledRidesCount> Registrator;

    public:
        TCompiledRidesCount(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig)
            : TBase(fetcherType, entityType, context, iteratorConfig)
        {
            const auto config = TBase::template GetConfigAs<TCompiledRidesIteratorConfig>();
            const auto& compiledSessionManager = context.GetServer()->GetDriveAPI()->GetMinimalCompiledRides();
            auto sinceEventId = TMaybe<ui64>();
            auto since = context.GetFetchInstant() - Yensured(config)->GetStartDeep();
            auto until = Now();
            auto tx = compiledSessionManager.BuildSession(true);
            while (true) {
                auto optionalCompiledSessions = compiledSessionManager.GetEvents<TMinimalCompiledRiding>(sinceEventId, { since, until }, tx, NSQL::TQueryOptions(100000));
                if (!optionalCompiledSessions) {
                    tx.Check();
                }
                if (optionalCompiledSessions->empty()) {
                    break;
                }
                for (auto&& compiledSession : *optionalCompiledSessions) {
                    TString key;
                    switch (entityType) {
                    case EAlertEntityType::Car:
                        key = compiledSession.GetObjectId();
                        break;
                    case EAlertEntityType::User:
                        key = compiledSession.GetHistoryUserId();
                        break;
                    default:
                        ythrow yexception() << entityType << " is not supported";
                    }
                    sinceEventId = compiledSession.GetHistoryEventId();
                    if (compiledSession.GetStartInstant() >= context.GetFetchInstant() - config->GetFinishDeep()) {
                        continue;
                    }
                    Sessions[key].insert(compiledSession.GetSessionId());
                }
            }
        }

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::SessionsCount;
        }

        bool ExtractData(TFetchedValue& data) const override {
            auto key = TBase::GetObjectId();

            ui64 sessionsCount = 0;
            auto p = Sessions.find(key);
            if (p != Sessions.end()) {
                sessionsCount = p->second.size();
            }

            data = sessionsCount;
            return true;
        }

    private:
        TMap<TString, TSet<TString>> Sessions;
    };
}
