#include "compiled_bill.h"

#include <drive/backend/billing/interfaces/history.h>
#include <drive/backend/billing/manager.h>

namespace NAlerts {
    IFetchedIterator::TFactory::TRegistrator<TSpentSumIterator> TSpentSumIterator::Registrator(EFetchedItems::SpentSum);

    bool TSpentSumIterator::ExtractData(TFetchedValue& data) const {
        const TInstant now = Now();
        auto config = TBase::template GetConfigAs<TSpentSumIteratorConfig>();
        if (!config) {
            return false;
        }
        const TInstant since = now - config->GetStart();
        const TInstant until = now - config->GetEnd();
        data = 0;
        TString userId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        const auto& billingManager = TBase::Context.GetServer()->GetDriveAPI()->GetBillingManager();
        auto session = billingManager.BuildSession(true);
        auto bills = billingManager.GetCompiledBills().GetUserBillsFromDB(userId, since, until, session);
        if (!bills) {
            ERROR_LOG << "TSpentSumIterator: Cannot retrieve compiled bills for user " << userId << " session error: " << session.GetStringReport() << Endl;
            return false;
        }
        TSet<TString> sessionIds;
        for (auto&& bill : *bills) {
            sessionIds.emplace(bill.GetSessionId());
        }
        if (config->GetIncludeActive()) {
            auto billingTasks = billingManager.GetActiveTasksManager().GetUsersTasks(NContainer::Scalar(userId), session);
            if (!billingTasks) {
                ERROR_LOG << "TSpentSumIterator: Cannot retrieve compiled bills for user " << userId << " session error: " << session.GetStringReport() << Endl;
                return false;
            }
            for (auto&& billingTask : *billingTasks) {
                sessionIds.emplace(billingTask.GetId());
            }
        }
        TMap<TString, TCachedPayments> payments;
        if (!billingManager.GetPaymentsManager().GetPayments(payments, MakeVector(sessionIds), session)) {
            ERROR_LOG << "TSpentSumIterator: Cannot retrieve payments for user " << userId << " session error: " << session.GetStringReport() << Endl;
            return false;
        }
        for (auto&& [_, cachedPayment] : payments) {
            for (auto&& paymentTask : cachedPayment.GetPayments()) {
                if (paymentTask.GetCreatedAt() >= since && paymentTask.GetCreatedAt() < until
                    && (config->GetBillingTypes().empty() || config->GetBillingTypes().contains(paymentTask.GetBillingType()))
                    && (config->GetTrustStatuses().empty() || config->GetTrustStatuses().contains(paymentTask.GetStatus()))) {
                    data += paymentTask.GetSum();
                }
            }
        }
        return true;
    }

    bool TSpentSumIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return NJson::ParseField(json, "billing_type", BillingTypes, false)
            && NJson::ParseField(json, "trust_status", TrustStatuses, false)
            && NJson::ParseField(json, "include_active", IncludeActive, false)
            && NJson::ParseField(json, "start", Start, true)
            && NJson::ParseField(json, "end", End, false);
    }

    NJson::TJsonValue TSpentSumIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "billing_type", BillingTypes);
        NJson::InsertField(result, "trust_status", TrustStatuses);
        NJson::InsertField(result, "include_active", IncludeActive);
        NJson::InsertField(result, "start", Start);
        NJson::InsertField(result, "end", End);
        return result;
    }

    NDrive::TScheme TSpentSumIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSVariants>("billing_type", "Тип списаний (терминал)").InitVariants<EBillingType>().SetMultiSelect(true);
        scheme.Add<TFSVariants>("trust_status", "Статусы платежа").InitVariants<NDrive::NTrustClient::EPaymentStatus>().SetMultiSelect(true);
        scheme.Add<TFSDuration>("start", "Глубина проверки, от");
        scheme.Add<TFSDuration>("end", "Глубина проверки, до");
        scheme.Add<TFSBoolean>("include_active", "Учитывать активные списания");
        return scheme;
    }
}
