#include "dictionary_tags.h"

namespace NAlerts {
    IFetchedIterator::TFactory::TRegistrator<TDictionaryTagFieldIterator> TDictionaryTagFieldIterator::Registrator(EFetchedItems::DictionaryTag);

    bool TDictionaryTagFieldIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "tag_name", TagName, true) &&
            NJson::ParseField(json, "field_name", FieldName, true) &&
            NJson::ParseField(json, "field_values", FieldValues, true);
    }

    NJson::TJsonValue TDictionaryTagFieldIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "tag_name", TagName);
        NJson::InsertField(result, "field_name", FieldName);
        NJson::InsertField(result, "field_values", FieldValues);
        return result;
    }

    NDrive::TScheme TDictionaryTagFieldIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("tag_name", "Имя тега").SetRequired(true);
        scheme.Add<TFSString>("field_name", "Имя поля").SetRequired(true);
        scheme.Add<TFSArray>("field_values", "Возможные значения (наличие поля при пустом множестве)").SetElement<TFSString>();
        return scheme;
    }

    bool TDictionaryTagFieldIterator::InitByObjects(IFetchedIterator& objectIterator) {
        auto config = TBase::template GetConfigAs<TDictionaryTagFieldIteratorConfig>();
        if (!config) {
            return false;
        }
        TSet<TString> objectIds;
        for (/*objectIterator*/; !objectIterator.IsFinished(); objectIterator.Next()) {
            auto object = objectIterator.GetObjectId();
            objectIds.emplace(object.Data(), object.Size());
        }
        const TFetcherContext& context = TBase::Context;
        auto tagEntity = NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
        if (!tagEntity.Defined()) {
            context.AddError("TDictionaryTagFieldIterator", TStringBuilder() << "Bad entity type " << ToString(GetEntityType()) << " for TDictionaryTagFieldIterator of fetcher " << ToString(GetFetcherName()));
            return false;
        }
        IEntityTagsManager::TOptionalTags tags;
        {
            const auto& tagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
            auto session = tagsManager.BuildSession(true);
            tags = tagsManager.RestoreTags(objectIds, { config->GetTagName() }, session);
            if (!tags) {
                context.AddError("TDictionaryTagFieldIterator", TStringBuilder() << "Could not restore tags in TDictionaryTagFieldIterator of fetcher " << ToString(GetFetcherName()) << " error: " << session.GetStringReport());
                return false;
            }
        }

        for (auto&& tag : *tags) {
            if (IDictionaryTag* dictionaryTagData = tag.MutableTagAs<IDictionaryTag>()) {
                auto fieldValue = dictionaryTagData->GetField(config->GetFieldName());
                if (fieldValue.Defined() && (config->GetFieldValues().empty() || config->GetFieldValues().contains(*fieldValue))) {
                    ObjectFieldsCount[tag.GetObjectId()] += 1;
                }
            } else {
                context.AddError("TDictionaryTagFieldIterator", TStringBuilder() << "Tag " << config->GetTagName() << " is not dictionary tag");
                return false;
            }
        }
        return true;
    }

    bool TDictionaryTagFieldIterator::ExtractData(TFetchedValue& data) const {
        TString objectId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        auto it = ObjectFieldsCount.find(objectId);
        if (it != ObjectFieldsCount.end()) {
            data = it->second;
        }
        return true;
    }
}
