#include "event_tags.h"

namespace NAlerts {
    IFetchedIterator::TFactory::TRegistrator<TEventTagEventCountIterator> TEventTagEventCountIterator::Registrator(EFetchedItems::EventTagEventCount);
    IFetchedIterator::TFactory::TRegistrator<TEventTagPresentEventsIterator> TEventTagPresentEventsIterator::Registrator(EFetchedItems::EventTagPresentEvents);
    IFetchedIterator::TFactory::TRegistrator<TEventTagMetaCountIterator> TEventTagMetaCountIterator::Registrator(EFetchedItems::EventTagMetaCount);

    bool TEventTagConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "tag_name", TagName, true) &&
            NJson::ParseField(json, "event_names", EventNames, true);
    }

    NJson::TJsonValue TEventTagConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "tag_name", TagName);
        NJson::InsertField(result, "event_names", EventNames);
        return result;
    }

    NDrive::TScheme TEventTagConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("tag_name", "Имя тега").SetRequired(true);
        scheme.Add<TFSArray>("event_names", "Имена событий").SetElement<TFSString>();
        return scheme;
    }

    bool IEventTagIterator::ExtractData(TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TEventTagConfig>();
        if (!config) {
            return false;
        }
        const TFetcherContext& context = TBase::Context;

        TString objectId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());

        auto tagEntity = NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
        if (!tagEntity.Defined()) {
            context.AddError("IEventTagIterator", TStringBuilder() <<
                                "Bad entity type " << ToString(GetEntityType()) << " for IEventTagIterator of fetcher " << ToString(GetFetcherName()));
            return false;
        }
        const auto& tagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
        auto session = tagsManager.BuildSession(true);
        auto tags = tagsManager.RestoreEntityTags(objectId, { config->GetTagName() }, session);
        if (!tags) {
            context.AddError("IEventTagIterator", TStringBuilder() <<
                                "Could not restore tags for object " << objectId << " in IEventTagIterator of fetcher " << ToString(GetFetcherName()));
            return false;
        }
        if (tags->empty()) {
            return true;
        }
        auto result = ProcessEventTags(*tags);
        if (!result) {
            context.AddError("IEventTagIterator", result.GetError());
            return false;
        }
        data = *result;
        return true;
    }

    TExpected<TFetchedValue, TString> TEventTagEventCountIterator::ProcessEventTags(const TVector<TDBTag>& tags) const {
        auto config = TBase::template GetConfigAs<TEventTagConfig>();
        if (!config) {
            return MakeUnexpected<TString>("Config has wrong type");
        }
        TFetchedValue data = 0;
        auto eventNamesSet = MakeSet(config->GetEventNames());
        for (auto&& tag : tags) {
            auto eventTag = tag.GetTagAs<IEventTag>();
            if (!eventTag) {
                return MakeUnexpected<TString>("Tag " + tag->GetName() + " is not EventTag");
            }
            data += eventTag->GetEventCount(eventNamesSet);
        }
        return data;
    }

    bool TEventTagPresentEventsConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "check_order", CheckOrder, true) &&
            TEventTagConfig::DeserializeFromJson(json);
    }

    NJson::TJsonValue TEventTagPresentEventsConfig::SerializeToJson() const {
        NJson::TJsonValue result = TEventTagConfig::SerializeToJson();
        NJson::InsertField(result, "check_order", CheckOrder);
        return result;
    }

    NDrive::TScheme TEventTagPresentEventsConfig::GetScheme(const IServerBase& server) const {
        NDrive::TScheme scheme = TEventTagConfig::GetScheme(server);
        scheme.Add<TFSBoolean>("check_order", "Проверять порядок").SetDefault(false);
        return scheme;
    }

    TExpected<TFetchedValue, TString> TEventTagPresentEventsIterator::ProcessEventTags(const TVector<TDBTag>& tags) const {
        auto config = TBase::template GetConfigAs<TEventTagPresentEventsConfig>();
        if (!config) {
            return MakeUnexpected<TString>("Config has wrong type");
        }
        TFetchedValue data = 0;
        for (auto&& tag : tags) {
            auto eventTag = tag.GetTagAs<IEventTag>();
            if (!eventTag) {
                return MakeUnexpected<TString>("Tag " + tag->GetName() + " is not EventTag");
            }
            bool hasEvents = false;
            if (config->GetCheckOrder()) {
                hasEvents = eventTag->HasEventsOrdered(config->GetEventNames());
            } else {
                hasEvents = eventTag->HasEvents(MakeSet(config->GetEventNames()));
            }
            if (hasEvents) {
                ++data;
            }
        }
        return data;
    }

    bool TEventTagMetaCountConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "field_path", FieldPath, true) &&
            NJson::ParseField(json, "ignore_empty", IgnoreEmpty, true) &&
            TEventTagConfig::DeserializeFromJson(json);
    }

    NJson::TJsonValue TEventTagMetaCountConfig::SerializeToJson() const {
        NJson::TJsonValue result = TEventTagConfig::SerializeToJson();
        NJson::InsertField(result, "ignore_empty", IgnoreEmpty);
        NJson::InsertField(result, "field_path", FieldPath);
        return result;
    }

    NDrive::TScheme TEventTagMetaCountConfig::GetScheme(const IServerBase& server) const {
        NDrive::TScheme scheme = TEventTagConfig::GetScheme(server);
        scheme.Add<TFSBoolean>("ignore_empty", "Не учитывать пустые значения").SetDefault(false);
        scheme.Add<TFSString>("field_path", "Путь до поля");
        return scheme;
    }

    TExpected<TFetchedValue, TString> TEventTagMetaCountIterator::ProcessEventTags(const TVector<TDBTag>& tags) const {
        auto config = TBase::template GetConfigAs<TEventTagMetaCountConfig>();
        if (!config) {
            return MakeUnexpected<TString>("Config has wrong type");
        }
        TFetchedValue data = 0;
        TSet<TString> fieldValues;
        bool countedEmpty = false;
        for (auto&& tag : tags) {
            auto eventTag = tag.GetTagAs<IEventTag>();
            if (!eventTag) {
                return MakeUnexpected<TString>("Tag " + tag->GetName() + " is not EventTag");
            }
            for (auto&& event : eventTag->GetEvents()) {
                NJson::TJsonValue value;
                if (event.GetMeta().IsDefined() && event.GetMeta().GetValueByPath(config->GetFieldPath(), value)) {
                    const TString& foundValue = value.GetStringRobust();
                    if (!fieldValues.contains(foundValue)) {
                        ++data;
                        fieldValues.emplace(foundValue);
                    }
                } else if (!config->GetIgnoreEmpty() && !countedEmpty) {
                    countedEmpty = true;
                    ++data;
                }
            }
        }
        return data;
    }
}
