#include "history_rides.h"

bool TLastHistoryRideIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
    return NJson::ParseField(json, "history_depth", HistoryDepth, true);
}

NJson::TJsonValue TLastHistoryRideIteratorConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "history_depth", HistoryDepth);
    return result;
}

NDrive::TScheme TLastHistoryRideIteratorConfig::GetScheme(const IServerBase& /* server */) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSDuration>("history_depth", "Максимальный возраст поездки");
    return scheme;
}

bool TLastHistoryRideIterator::ExtractData(NAlerts::TFetchedValue& data) const {
    const auto& server = *Yensured(Context.GetServer());
    const auto& driveApi = *Yensured(server.GetDriveAPI());
    const auto& config = *GetConfigAs<TLastHistoryRideIteratorConfig>();

    THistoryRidesContext context(server, TInstant::Now() - config.GetHistoryDepth());
    auto tx = Yensured(server.GetDriveAPI())->BuildTx<NSQL::ReadOnly>();
    auto ydbTx = driveApi.BuildYdbTx<NSQL::ReadOnly>("history_rides_iterator", &server);

    TString objectId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
    switch (GetEntityType()) {
        case NAlerts::EAlertEntityType::Car:
            if (!context.InitializeCars({objectId}, tx, ydbTx)) {
                return false;
            }
            break;
        case NAlerts::EAlertEntityType::User:
            if (!context.InitializeUser(objectId, tx, ydbTx)) {
                return false;
            }
            break;
        default:
            return false;
    }

    THistoryRideObject ride;
    TInstant lastTs = TInstant::Zero();
    for (auto it = context.GetIterator(); it.GetAndNext(ride);) {
        if (ride.IsActive()) {
            continue;
        }

        const auto rideLastTs = ride.GetLastTS();
        if (rideLastTs > lastTs) {
            lastTs = rideLastTs;
        }
    }

    data = (Now() - lastTs).MicroSeconds();
    return true;
}

NAlerts::IIteratorConfig::TFactory::TRegistrator<TLastHistoryRideIteratorConfig> TLastHistoryRideIteratorConfig::Registrator(NAlerts::EFetchedItems::LastHistoryRide);
NAlerts::IFetchedIterator::TFactory::TRegistrator<TLastHistoryRideIterator> TLastHistoryRideIterator::Registrator(NAlerts::EFetchedItems::LastHistoryRide);
