#include "iterators.h"

#include <drive/backend/alerts/condition.h>
#include <drive/backend/alerts/fetchers/interfaces.h>


namespace NAlerts {
    template <class TFetcher>
    class TContainerIteratorImpl : public IFetchedIterator {
    public:
        TContainerIteratorImpl(const EAlertEntityType entityType, const typename TFetcher::TContainerType& objects, const TContainerIteratorBase& owner)
            : IFetchedIterator(EDataFetcherType::ETestFetcher, entityType)
            , Owner(owner)
            , Data(IFetchedIterator::InvalidData())
            , Current(objects.begin())
            , End(objects.end())
        {
            while (Current != End && !Owner.GetContext().GetShardingPolicy().CheckMatching(TString(GetObjectId()))) {
                ++Current;
            }
        }

        TStringBuf GetObjectId() const final {
            return TFetcher::GetObjectId(Current, GetEntityType());
        }

        TStringBuf GetObjectId(const EAlertEntityType entityType) const final {
            return TFetcher::GetObjectId(Current, entityType);
        }

        TMaybe<TFetchedValue> Get() final {
            if (Data == IFetchedIterator::InvalidData()) {
                if (!Owner.ExtractData(Data)) {
                    return {};
                }
            }
            return Data;
        }

        bool IsFinished() const override {
            return Current == End;
        }

        bool Next() final {
            if (IsFinished()) {
                return false;
            }
            Data = IFetchedIterator::InvalidData();
            ++Current;
            for (; Current != End; ++Current) {
                if (!Owner.GetContext().GetShardingPolicy().CheckMatching(TString(GetObjectId()))) {
                    continue;
                }
                return true;
            }
            return false;
        }

        bool Init(const IServiceDataFetcher&) override {
            return false;
        }

        EFetchedItems GetField() const override {
            return EFetchedItems::Main;
        }

        const typename TFetcher::TContainerType::const_iterator& GetCurrent() const {
            CHECK_WITH_LOG(!IsFinished());
            return Current;
        }

        IDataIntervalChecker::TPtr GetChecker() const override {
            return Owner.GetChecker();
        }

        TString GetSubstitutionName() const override {
            return Owner.GetSubstitutionName();
        }

    private:
        const TContainerIteratorBase& Owner;
        TFetchedValue Data;
    protected:
        typename TFetcher::TContainerType::const_iterator Current;
        typename TFetcher::TContainerType::const_iterator End;
    };

    THolder<IFetchedIterator> TContainerIteratorBase::GetObjectIterator(const IServiceDataFetcher& fetcher) const {
        if (auto setData = dynamic_cast<const IIdsDataFetcher*>(&fetcher)) {
            return MakeHolder<TContainerIteratorImpl<IIdsDataFetcher>>(fetcher.GetEntityType(), setData->GetObjects(), *this);
        } else if (auto userFetcher = dynamic_cast<const IUserDataFetcher*>(&fetcher)) {
            return MakeHolder<TContainerIteratorImpl<IUserDataFetcher>>(fetcher.GetEntityType(), userFetcher->GetObjects(), *this);
        } else if (auto sessionsFetcher = dynamic_cast<const ISessionsDataFetcher*>(&fetcher)) {
            return MakeHolder<TContainerIteratorImpl<ISessionsDataFetcher>>(fetcher.GetEntityType(), sessionsFetcher->GetObjects(), *this);
        }
        return nullptr;
    }

    bool TContainerIteratorBase::Init(const IServiceDataFetcher& fetcher) {
        auto objectIterator = GetObjectIterator(fetcher);
        if (objectIterator) {
            if (!InitByObjects(*objectIterator)) {
                return false;
            }
            Impl.Reset(GetObjectIterator(fetcher));
            if (Impl) {
                return true;
            }
        }
        return false;
    }

    template <>
    const TDriveUserData* TContainerIteratorBase::GetDataCurrent<TDriveUserData>() const {
        CHECK_WITH_LOG(Impl);
        auto iterator = dynamic_cast<const TContainerIteratorImpl<IUserDataFetcher>*>(Impl.Get());
        if (iterator) {
            return &IUserDataFetcher::GetObjectData(iterator->GetCurrent());
        }
        return nullptr;
    }

    template <>
    const ::IEventsSession<TCarTagHistoryEvent>* TContainerIteratorBase::GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>() const {
        CHECK_WITH_LOG(Impl);
        auto iterator = dynamic_cast<const TContainerIteratorImpl<ISessionsDataFetcher>*>(Impl.Get());
        if (iterator) {
            return ISessionsDataFetcher::GetObjectData(iterator->GetCurrent()).Get();
        }
        return nullptr;
    }

    IFetchedIterator::TFactory::TRegistrator<TMainContainerIterator> TMainContainerIterator::Registrator(EFetchedItems::Main);
    IFetchedIterator::TFactory::TRegistrator<TLandingsIterator> TLandingsIterator::Registrator(EFetchedItems::LandingAcceptanceDelta);
    IFetchedIterator::TFactory::TRegistrator<TActionTimeIterator> TActionTimeIterator::Registrator(EFetchedItems::LastActionDelta);

    bool TTimeIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_DURATION_OPT(json, "history_deep", HistoryDeep);
        TJsonProcessor::ReadContainer(json, "history_actions", HistoryActions);
        TJsonProcessor::ReadContainer(json, "history_tags", HistoryTags);
        return true;
    }

    NJson::TJsonValue TTimeIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        TJsonProcessor::WriteDurationString(result, "history_deep", HistoryDeep);
        TJsonProcessor::WriteContainerArrayStrings(result, "history_actions", HistoryActions);
        TJsonProcessor::WriteContainerArray(result, "history_tags", HistoryTags);
        return result;
    }

    NDrive::TScheme TTimeIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSDuration>("history_deep", "Глубина проверки истории").SetDefault(TDuration::Days(60));
        scheme.Add<TFSVariants>("history_actions", "Фильтр действий").InitVariants<EObjectHistoryAction>().SetMultiSelect(true);
        scheme.Add<TFSArray>("history_tags", "Теги для проверки действий").SetElement<TFSString>();
        return scheme;
    }

    bool TLandingsIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        TJsonProcessor::ReadContainer(json, "landings", LandingsFilter);
        return true;

    }

    NJson::TJsonValue TLandingsIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        TJsonProcessor::WriteContainerArray(result, "landings", LandingsFilter);
        return result;
    }

    NDrive::TScheme TLandingsIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSArray>("landings", "Лендинги для проверки").SetElement<TFSString>();
        return scheme;
    }

    TActionTimeIterator::TActionTimeIterator(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig)
        : TBase(fetcherType, entityType, context, iteratorConfig)
    {
        auto config = TBase::template GetConfigAs<TTimeIteratorConfig>();
        Y_ENSURE_BT(config);
        auto tagEntity = NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
        if (!tagEntity.Defined()) {
            ERROR_LOG << "Bad entity type " << ToString(GetEntityType()) << " for TActionTimeIterator of fetcher " << ToString(fetcherType) << Endl;
            return;
        }
        const auto& tagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
        auto session = tagsManager.BuildSession(true);
        auto since = context.GetFetchInstant() - config->GetHistoryDeep();
        auto until = TMaybe<TTagHistoryEventId>();
        auto limit = size_t(100000);
        auto queryOptions = IEntityTagsManager::TQueryOptions(limit, /*descending=*/true)
            .SetActions(config->GetHistoryActions())
            .SetTags(config->GetHistoryTags())
        ;
        while (true) {
            if (!until) {
                INFO_LOG << entityType << ": fetch " << JoinSeq(",", config->GetHistoryTags()) << " since " << since << Endl;
            }
            auto events = tagsManager.GetEvents({0, until}, since, session, queryOptions);
            Y_ENSURE_BT(events, session.GetStringReport());
            for (auto&& ev : *events) {
                auto eventId = ev.GetHistoryEventId();
                until = std::min(until.GetOrElse(eventId), eventId);
                const auto& objectId = ev.GetObjectId();
                if (Results.contains(objectId)) {
                    continue;
                }
                Results[objectId] = context.GetFetchInstant() - ev.GetHistoryInstant();
            }
            if (events->size() < limit) {
                break;
            }
            NOTICE_LOG << entityType << ": refetch " << JoinSeq(",", config->GetHistoryTags()) << " since " << since << " until " << until << Endl;
        }
    }
}
