#pragma once

#include <drive/backend/alerts/condition.h>
#include <drive/backend/alerts/fetcher.h>

#include <drive/backend/database/drive_api.h>
#include <drive/backend/database/drive/landing.h>
#include <drive/backend/tags/tags_manager.h>

#include <util/generic/adaptor.h>

namespace NAlerts {

    class TContainerIteratorBase : public IFetchedIterator { // TODO: add allowed entity types on deserialization
    public:
        TContainerIteratorBase(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig)
            : IFetchedIterator(fetcherType, entityType)
            , Context(context)
            , Config(iteratorConfig)
        {}

        TMaybe<TFetchedValue> Get() override {
            CHECK_WITH_LOG(Impl);
            return Impl->Get();
        }

        TStringBuf GetObjectId() const override {
            CHECK_WITH_LOG(Impl);
            return Impl->GetObjectId();
        }

        TStringBuf GetObjectId(const EAlertEntityType entityType) const override {
            CHECK_WITH_LOG(Impl);
            return Impl->GetObjectId(entityType);
        }

        bool IsFinished() const override {
            CHECK_WITH_LOG(Impl);
            return Impl->IsFinished();
        }

        virtual bool Next() final {
            CHECK_WITH_LOG(Impl);
            return Impl->Next();
        }

        const TFetcherContext& GetContext() const {
            return Context;
        }

        TString GetStringId() const {
            CHECK_WITH_LOG(Impl);
            auto objectId = Impl->GetObjectId();
            return TString(objectId.Data(), objectId.Size());
        }

        THolder<IFetchedIterator> GetObjectIterator(const IServiceDataFetcher& fetcher) const;
        virtual bool Init(const IServiceDataFetcher& fetcher) final;
        virtual bool ExtractData(TFetchedValue& data) const = 0;
        virtual bool InitByObjects(IFetchedIterator& /*iterator*/) {
            return true;
        }

        virtual IDataIntervalChecker::TPtr GetChecker() const override {
            return Config ? Config->GetChecker() : nullptr;
        }

        virtual TString GetSubstitutionName() const override {
            return GetChecker() ? GetChecker()->GetIteratorName() : "";
        }

    protected:
        template<class T>
        const T* GetConfigAs() const {
            return dynamic_cast<const T*>(Config.Get());
        }

        template <class TIteratorDataType>
        const TIteratorDataType* GetDataCurrent() const {
            ERROR_LOG << "Not implemented fetcher data" << Endl;
            return nullptr;
        }

        const TFetcherContext& Context;
    private:
        IIteratorConfig::TPtr Config;
        THolder<IFetchedIterator> Impl;
    };

    class TMainContainerIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TMainContainerIterator> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::Main;
        }
    private:
        virtual bool ExtractData(TFetchedValue& data) const override {
            data = 1;
            return true;
        }
    };

    class TEmptyConfig: public IIteratorConfig {
        using TBase = IIteratorConfig;

    public:
        using TBase::TBase;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override {
            Y_UNUSED(json);
            return true;
        }

        virtual NJson::TJsonValue SerializeToJson() const override {
            return NJson::JSON_MAP;
        }

        virtual NDrive::TScheme GetScheme(const IServerBase& server) const override {
            Y_UNUSED(server);
            return NDrive::TScheme();
        }
    };

    class TLandingsIteratorConfig : public IIteratorConfig {
        R_READONLY(TSet<TString>, LandingsFilter);
        using TBase = IIteratorConfig;

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TLandingsIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TLandingsIterator> Registrator;

    public:
        using TBase::TBase;
        virtual EFetchedItems GetField() const override {
            return EFetchedItems::LandingAcceptanceDelta;
        }

        bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TLandingsIteratorConfig>();
            if (!config) {
                return false;
            }

            const TFetcherContext& context = TBase::Context;
            if (config->GetLandingsFilter().empty()) {
                return true;
            }

            TString userId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
            auto gAcceptances = context.GetServer()->GetDriveAPI()->GetUserLandingData()->FetchInfo(userId, context.GetFetchInstant());
            auto result = gAcceptances.GetResultPtr(userId);
            if (!result) {
                return true;
            }
            for (auto&& acceptance : result->GetLandings()) {
                if (config->GetLandingsFilter().contains(acceptance.GetId())) {
                    data = (context.GetFetchInstant() - acceptance.GetLastAcceptedAt()).Seconds();
                    return true;
                }
            }
            return true;
        }
    };

    class TTimeIteratorConfig : public IIteratorConfig {
        R_READONLY(TSet<EObjectHistoryAction>, HistoryActions);
        R_READONLY(TDuration, HistoryDeep, TDuration::Days(30));
        R_READONLY(TSet<TString>, HistoryTags);
        using TBase = IIteratorConfig;

    public:
        using TBase::TBase;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TActionTimeIterator : public TContainerIteratorBase {
    private:
        using TBase = TContainerIteratorBase;

    private:
        static IFetchedIterator::TFactory::TRegistrator<TActionTimeIterator> Registrator;

    public:
        TActionTimeIterator(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig);

        using TBase::GetObjectId;
        using TBase::Context;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::LastActionDelta;
        }

        bool ExtractData(TFetchedValue& data) const override {
            const auto& objectId = GetObjectId();
            auto p = Results.find(objectId);
            if (p != Results.end()) {
                data = p->second.Seconds();
            }
            return true;
        }

    private:
        THashMap<TString, TDuration> Results;
    };
}
