#include "location.h"

#include <drive/backend/database/config.h>
#include <drive/backend/device_snapshot/manager.h>
#include <drive/backend/drivematics/zone/zone.h>
#include <drive/backend/drivematics/zone/config/zone_config.h>

#include <rtline/library/geometry/polyline.h>

bool TLocationIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& value) {
    return
        NJson::ParseField(value["included_area_ids"], IncludedAreaIds) &&
        NJson::ParseField(value["excluded_area_ids"], ExcludedAreaIds) &&
        NJson::ParseField(value["included_location_tags"], IncludedLocationTags) &&
        NJson::ParseField(value["excluded_location_tags"], ExcludedLocationTags) &&
        NJson::ParseField(value["included_group_ids"], IncludedZoneIds) &&
        NJson::ParseField(value["excluded_group_ids"], ExcludedZoneIds) &&
        NJson::ParseField(value["logging_enabled"], LoggingEnabled);
}

NJson::TJsonValue TLocationIteratorConfig::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "included_area_ids", IncludedAreaIds);
    NJson::InsertNonNull(result, "excluded_area_ids", ExcludedAreaIds);
    NJson::InsertNonNull(result, "included_location_tags", IncludedLocationTags);
    NJson::InsertNonNull(result, "excluded_location_tags", ExcludedLocationTags);
    NJson::InsertNonNull(result, "included_group_ids", IncludedZoneIds);
    NJson::InsertNonNull(result, "excluded_group_ids", ExcludedZoneIds);
    NJson::InsertField(result, "logging_enabled", LoggingEnabled);
    return result;
}

NDrive::TScheme TLocationIteratorConfig::GetScheme(const IServerBase& server) const {
    const auto& areaManager = server.GetAsSafe<NDrive::IServer>().GetDriveDatabase().GetAreaManager();
    const auto areaIds = areaManager.GetAreaIds();
    const auto locationTags = areaManager.GetAreaTags();
    NDrive::TScheme result;
    result.Add<TFSVariants>("included_area_ids").SetVariants(areaIds).SetMultiSelect(true);
    result.Add<TFSVariants>("excluded_area_ids").SetVariants(areaIds).SetMultiSelect(true);
    result.Add<TFSVariants>("included_location_tags").SetVariants(locationTags).SetMultiSelect(true);
    result.Add<TFSVariants>("excluded_location_tags").SetVariants(locationTags).SetMultiSelect(true);
    result.Add<TFSBoolean>("logging_enabled").SetDefault(LoggingEnabled);

    if (server.GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetConfig().GetZoneConfig()->GetEnable()) {
        NDrive::TFSVariants::TCompoundVariants compoundVariant;
        auto action = [&compoundVariant](const NDrivematics::TZone& zone) -> void {
            compoundVariant.emplace(
                NDrive::TFSVariants::TCompoundVariant{zone.GetInternalId(), zone.GetName(), zone.GetOwnerDef("")}
            );
        };
        Y_UNUSED(server.GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetZoneDB()->ForObjectsList(action, TInstant::Now()));
        result.Add<TFSVariable>("included_group_ids").MutableCondition().SetCompoundVariants(compoundVariant).SetMultiSelect(true);
        result.Add<TFSVariable>("excluded_group_ids").MutableCondition().SetCompoundVariants(compoundVariant).SetMultiSelect(true);
    }
    return result;
}

bool TLocationIterator::ExtractData(NAlerts::TFetchedValue& data) const {
    auto config = GetConfigAs<TLocationIteratorConfig>();
    Y_ENSURE(config);
    auto objectId = GetCarId();
    auto imei = Context.GetServer()->GetDriveAPI()->GetIMEI(objectId);
    if (!imei) {
        WARNING_LOG << "LocationIterator for " << objectId << ": no imei" << Endl;
        return false;
    }

    if (!Locations.Wait(WaitDeadline)) {
        ERROR_LOG << "LocationIterator: wait timeout" << Endl;
    }

    auto snapshot = Context.GetServer()->GetSnapshotsManager().GetSnapshot(objectId);
    auto snapshotLocation = snapshot.GetLocation(config->GetLocationName());
    auto storageLocation = Locations.GetValue().FindPtr(imei);

    auto location = storageLocation;
    auto source = TString{"storage"};
    if (
        (snapshotLocation && storageLocation && snapshotLocation->Timestamp > storageLocation->Timestamp) ||
        !location
    ) {
        location = snapshotLocation.Get();
        source = "snapshot";
    }
    if (!location) {
        WARNING_LOG << "LocationIterator for " << objectId << ": no location" << Endl;
        return false;
    }

    auto server = Context.GetServer();
    Y_ENSURE(server);
    const auto& areaManager = server->GetDriveDatabase().GetAreaManager();
    const auto coordinate = location->GetCoord();
    const auto areaIds = areaManager.GetAreaIdsInPoint(coordinate);
    NDrive::TZoneIds zoneIds;
    {
        if (Context.GetServer()->GetDriveAPI()->GetConfig().GetZoneConfig()->GetEnable()) {
            auto getZoneIds = [&](const NDrive::TZoneIds& ids) -> bool {
                return server->GetDriveAPI()->GetZoneDB()->GetZoneIdsInPoint(ids, coordinate, std::ref(zoneIds), areaManager);
            };
            if (config->HasIncludedZoneIds()) {
                R_ENSURE(getZoneIds(config->GetIncludedZoneIdsRef()), {},"can't get IncludedZoneIds in point");
            }
            if (config->HasExcludedZoneIds()) {
                R_ENSURE(getZoneIds(config->GetExcludedZoneIdsRef()), {},"can't get ExcludedZoneIds in point");
            }
        }
    }
    const auto locationTags = areaManager.GetTagsInPoint(coordinate);
    if (config->HasIncludedAreaIds()) {
        if (!HasIntersection(config->GetIncludedAreaIdsRef(), areaIds)) {
            data = 0;
            return true;
        }
    }
    if (config->HasExcludedAreaIds()) {
        if (HasIntersection(config->GetExcludedAreaIdsRef(), areaIds)) {
            data = 0;
            return true;
        }
    }
    if (config->HasIncludedZoneIds()) {
        if (!HasIntersection(config->GetIncludedZoneIdsRef(), zoneIds)) {
            data = 0;
            return true;
        }
    }
    if (config->HasExcludedZoneIds()) {
        if (HasIntersection(config->GetExcludedZoneIdsRef(), zoneIds)) {
            data = 0;
            return true;
        }
    }
    if (config->HasIncludedLocationTags()) {
        if (!HasIntersection(config->GetIncludedLocationTagsRef(), locationTags)) {
            data = 0;
            return true;
        }
    }
    if (config->HasExcludedLocationTags()) {
        if (HasIntersection(config->GetExcludedLocationTagsRef(), locationTags)) {
            data = 0;
            return true;
        }
    }

    if (config->IsLoggingEnabled()) {
        NDrive::TEventLog::Log("LocationIteratorPassed", NJson::TMapBuilder
            ("object_id", objectId)
            ("imei", imei)
            ("location", NJson::ToJson(location))
            ("source", source)
            ("tags", NJson::ToJson(locationTags))
        );
    }

    data = 1;
    return true;
}

bool TLocationIterator::InitByObjects(IFetchedIterator& objectIterator) {
    auto config = GetConfigAs<TLocationIteratorConfig>();
    if (!config) {
        return false;
    }

    if (!Context.GetServer()->GetDriveAPI()->GetConfig().GetZoneConfig()->GetEnable() &&
            (config->HasIncludedZoneIds() || config->HasExcludedZoneIds())) {
        return false;
    }

    TVector<TString> objectIds;
    for (; !objectIterator.IsFinished(); objectIterator.Next()) {
        objectIds.push_back(ToString(objectIterator.GetObjectId()));
    }
    auto imeis = Context.GetServer()->GetDriveAPI()->GetIMEIs(objectIds);
    auto sensorClient = Context.GetServer()->GetSensorApi();
    auto timeout = config->GetLocationQueryTimeout();
    auto queryOptions = NDrive::ISensorApi::TLocationsQueryOptions();
    queryOptions
        .SetName(config->GetLocationName())
        .MutableFetchOptions()
            .SetTimeout(timeout)
    ;
    Locations = Yensured(sensorClient)->GetLocations(imeis, queryOptions);
    WaitDeadline = Now() + timeout;
    return true;
}

NAlerts::IIteratorConfig::TFactory::TRegistrator<TLocationIteratorConfig> TLocationIteratorConfig::Registrator(NAlerts::EFetchedItems::Location);
NAlerts::IFetchedIterator::TFactory::TRegistrator<TLocationIterator> TLocationIterator::Registrator(NAlerts::EFetchedItems::Location);
