#include "maintenance.h"

#include <drive/backend/cars/car.h>
#include <drive/telematics/api/sensor/history.h>
#include <drive/telematics/protocol/sensor.h>


namespace {
    bool InitByObjects(NAlerts::IFetchedIterator& iterator, TMap<TString, TMaintenanceInfo>& maintenanceInfo, const NAlerts::TFetcherContext& context, const TString& src) {
        maintenanceInfo.clear();
        TSet<TString> objectIds;
        for (/* iterator */; !iterator.IsFinished(); iterator.Next()) {
            auto object = iterator.GetObjectId();
            objectIds.emplace(object.Data(), object.Size());
        }
        auto session = context.GetServer()->GetDriveAPI()->GetMaintenanceDB().BuildSession(/* readOnly = */ true);
        auto cars = context.GetServer()->GetDriveAPI()->GetCarsData()->FetchInfo(objectIds, session);
        if (!cars) {
            context.AddError(src, TStringBuilder() << "Could not restore cars in " << src << " error: " << session.GetStringReport());
            return false;
        }
        TMap<TString, TString> vinsToId;
        Transform(cars.begin(), cars.end(), std::inserter(vinsToId, vinsToId.begin()), [](const auto& it) -> std::pair<TString, TString> { return { it.second.GetVin(), it.second.GetId() }; });
        auto infos = context.GetServer()->GetDriveAPI()->GetMaintenanceDB().GetObjects(MakeSet(NContainer::Keys(vinsToId)), session);
        if (!infos) {
            context.AddError(src, TStringBuilder() << "Could not restore maintenance info in " << src << " §error: " << session.GetStringReport());
            return false;
        }
        for (auto info : *infos) {
            if (auto id = vinsToId.FindPtr(info.GetVIN())) {
                maintenanceInfo.emplace(*id, std::move(info));
            }
        }
        return true;
    }
}

namespace NAlerts {
    IFetchedIterator::TFactory::TRegistrator<TMaintenanceTimeIterator> TMaintenanceTimeIterator::Registrator(EFetchedItems::MaintenanceTime);
    IFetchedIterator::TFactory::TRegistrator<TMaintenanceMileageDeltaIterator> TMaintenanceMileageDeltaIterator::Registrator(EFetchedItems::MaintenanceMileageDelta);

    TMaybe<NDrive::TSensor> TSensorFetcher::FetchSensor(const TFetcherContext& context, const TString& carId, const ui64 id) {
        TString imei = context.GetServer()->GetDriveAPI()->GetIMEI(carId);
        if (!imei) {
            return {};
        }

        const NDrive::TSensorId sensorId(id);
        try {
            const auto* sensorApi = context.GetServer()->GetSensorApi();
            if (!sensorApi) {
                return {};
            }
            auto sensorData = sensorApi->GetSensor(imei, sensorId, {}, 3).ExtractValueSync();
            return sensorData;
        } catch (const std::exception& e) {
            ERROR_LOG << "SensorFetcher::FetchSensor: " << FormatExc(e) << Endl;
            return {};
        }
    }

    bool TMaintenanceIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_BOOL_OPT(json, "reverse_value", Reverse);
        JREAD_BOOL_OPT(json, "only_active", OnlyOpen);
        JREAD_UINT_OPT(json, "mileage_sensor", MileageSensorId);
        return true;

    }

    NJson::TJsonValue TMaintenanceIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        JWRITE(result, "reverse_value", Reverse);
        JWRITE(result, "only_active", OnlyOpen);
        JWRITE(result, "mileage_sensor", MileageSensorId);
        return result;
    }

    NDrive::TScheme TMaintenanceIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSBoolean>("reverse_value", "Разница между ТО и текущим пробегом").SetDefault(false);
        scheme.Add<TFSBoolean>("only_active", "Только текущие").SetDefault(true);
        scheme.Add<TFSNumeric>("mileage_sensor", "Сенсор пробега").SetMin(0).SetDefault(2103);
        return scheme;
    }

    bool TMaintenanceTimeIterator::InitByObjects(IFetchedIterator& iterator) {
        return ::InitByObjects(iterator, MaintenanceInfo, TBase::Context, ToString(EFetchedItems::MaintenanceTime));
    }

    bool TMaintenanceMileageDeltaIterator::InitByObjects(IFetchedIterator& iterator) {
        return ::InitByObjects(iterator, MaintenanceInfo, TBase::Context, ToString(EFetchedItems::MaintenanceMileageDelta));
    }
}
