#pragma once

#include "iterators.h"

#include <drive/backend/major/client.h>

#include <drive/telematics/protocol/sensor.h>

namespace NAlerts {

    class TSensorFetcher {
    public:
        static TMaybe<NDrive::TSensor> FetchSensor(const TFetcherContext& context, const TString& carId, const ui64 sensorId);
    };

    class TMaintenanceIteratorConfig : public IIteratorConfig {
        using TBase = IIteratorConfig;
        R_READONLY(bool, Reverse, false);
        R_READONLY(bool, OnlyOpen, true);
        R_READONLY(ui32, MileageSensorId, 2103);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TMaintenanceTimeIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TMaintenanceTimeIterator> Registrator;
    public:
        using TBase::TBase;
        virtual EFetchedItems GetField() const override {
            return EFetchedItems::MaintenanceTime;
        }

        virtual TString GetCarId() const {
            return TString(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        }

        virtual bool InitByObjects(IFetchedIterator& iterator) override;
        bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TMaintenanceIteratorConfig>();
            if (!config) {
                return false;
            }
            const TFetcherContext& context = TBase::Context;
            auto mbInfo = MaintenanceInfo.FindPtr(GetCarId());
            if (!mbInfo || (config->IsOnlyOpen() && mbInfo->HasReadyDate())) {
                return false;
            }

            data = (context.GetFetchInstant() > mbInfo->GetStartDate()) ? (context.GetFetchInstant() - mbInfo->GetStartDate()).Seconds() : 0;
            return true;
        }

    private:
        TMap<TString, TMaintenanceInfo> MaintenanceInfo;
    };

    class TMaintenanceMileageDeltaIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TMaintenanceMileageDeltaIterator> Registrator;
    public:
        using TBase::TBase;
        virtual EFetchedItems GetField() const override {
            return EFetchedItems::MaintenanceMileageDelta;
        }

        virtual TString GetCarId() const {
            return TString(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        }

        virtual bool InitByObjects(IFetchedIterator& iterator) override;
        bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TMaintenanceIteratorConfig>();
            if (!config) {
                return false;
            }
            const TFetcherContext& context = TBase::Context;

            auto carId = GetCarId();
            auto mbInfo = MaintenanceInfo.FindPtr(carId);
            if (!mbInfo || (config->IsOnlyOpen() && mbInfo->HasReadyDate())) {
                return false;
            }

            if (!config->GetMileageSensorId()) {
                return false;
            }
            TMaybe<NDrive::TSensor> mileageSensor = TSensorFetcher::FetchSensor(context, carId, config->GetMileageSensorId());
            if (!mileageSensor) {
                return false;
            }

            double mileage = mileageSensor->ConvertTo<double>();

            if (config->IsReverse()) {
                data = (mbInfo->HasMileage() && mbInfo->GetMileageUnsafe() > mileage) ? mbInfo->GetMileageUnsafe() - mileage : 0;
            } else {
                if (mbInfo->HasMileage()) {
                    data = (mbInfo->GetMileageUnsafe() < mileage) ? mileage - mbInfo->GetMileageUnsafe() : 0;
                } else {
                    data = mileage;
                }
            }
            return true;
        }

    private:
        TMap<TString, TMaintenanceInfo> MaintenanceInfo;
    };
}
