#pragma once
#include "iterators.h"

#include <drive/library/cpp/user_events_api/client.h>

namespace NAlerts {
    class TRequestTimeIteratorConfig : public IIteratorConfig {
        R_READONLY(TDuration, HistoryDeep, TDuration::Max());
        R_READONLY(ui32, MaxRequestsCount, 100);
        R_READONLY(TSet<TString>, Sources);
        R_READONLY(TDuration, RequestTimeout, TDuration::Seconds(2));
        using TBase = IIteratorConfig;

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TRequestTimeIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TRequestTimeIterator> Registrator;
    public:
        using TBase::TBase;
        using TBase::GetObjectId;
        using TBase::Context;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::LastRequestTime;
        }

        bool ExtractData(TFetchedValue& data) const override {
            auto config = TBase::template GetConfigAs<TRequestTimeIteratorConfig>();
            if (!config) {
                return false;
            }
            const TFetcherContext& context = TBase::Context;

            if (!context.GetServer()->GetUserEventsApi()) {
                return false;
            }

            auto futureUserRequests = context.GetServer()->GetUserEventsApi()->GetUserRequests(TString(GetObjectId().Data(), GetObjectId().Size()), ((config->GetHistoryDeep() == TDuration::Max()) ? TInstant::Zero() : context.GetFetchInstant() - config->GetHistoryDeep()), TInstant::Max(), config->GetMaxRequestsCount());
            if (!futureUserRequests.Wait(config->GetRequestTimeout()) || !futureUserRequests.HasValue()) {
                return false;
            }

            TMaybe<TDuration> lastRequestTime;
            for (auto&& request : futureUserRequests.GetValue().GetRequests()) {
                if (!config->GetSources().contains(request.GetSource())) {
                    continue;
                }

                if (!lastRequestTime.Defined()) {
                    lastRequestTime = TDuration::Max();
                }
                lastRequestTime = Min(*lastRequestTime, (context.GetFetchInstant() > request.GetTimestamp()) ? context.GetFetchInstant() - request.GetTimestamp() : TDuration::Zero());
            }
            if (lastRequestTime.Defined()) {
                data = lastRequestTime->Seconds();
            }
            return true;
        }
    };
}
