#include "iterators.h"

#include <drive/telematics/api/sensor/history.h>
#include <drive/telematics/protocol/sensor.h>

namespace NAlerts {
    class TSensorIteratorConfig : public IIteratorConfig {
        using TBase = IIteratorConfig;
        R_READONLY(ui32, SensorId, 0);
        R_READONLY(TDuration, MinSensorAge, TDuration::Minutes(20));

    public:
        using TBase::TBase;
        bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        NJson::TJsonValue SerializeToJson() const override;
        NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    bool TSensorIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        JREAD_INT_OPT(json, "sensor_id", SensorId);
        JREAD_DURATION_OPT(json, "min_sensor_age", MinSensorAge);
        return true;
    }

    NJson::TJsonValue TSensorIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        JWRITE(result, "sensor_id", SensorId);
        JWRITE_DURATION(result, "min_sensor_age", MinSensorAge);
        return result;
    }

    NDrive::TScheme TSensorIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSNumeric>("sensor_id", "Идентификатор сенсора");
        scheme.Add<TFSNumeric>("min_sensor_age", "Минимальное время жизни сенсора");
        return scheme;
    }

    IIteratorConfig::TFactory::TRegistrator<TSensorIteratorConfig> RegistratorConfig(EFetchedItems::Sensor);

    class TSensorIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TSensorIterator> Registrator;

    public:
        using TBase::TBase;
        EFetchedItems GetField() const override {
            return EFetchedItems::Sensor;
        }

        virtual TString GetCarId() const {
            auto session = GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>();
            if (!session) {
                return TString(GetObjectId().Data(), GetObjectId().Size());
            }
            return session->GetObjectId();
        }

        bool InitByObjects(IFetchedIterator& objectIterator) override {
            auto config = TBase::template GetConfigAs<TSensorIteratorConfig>();
            if (!config || !config->GetSensorId()) {
                return false;
            }
            try {
                const auto* sensorApi = Context.GetServer()->GetSensorApi();
                if (!sensorApi) {
                    Context.AddError("TSensorIterator", "could not get sensor api");
                    return false;
                }
                TVector<TString> carIds;
                while (!objectIterator.IsFinished()) {
                    auto object = objectIterator.GetObjectId(EAlertEntityType::Car);
                    carIds.emplace_back(object.Data(), object.Size());
                    objectIterator.Next();
                }
                auto imeis = Context.GetServer()->GetDriveAPI()->GetIMEIs(carIds);
                NDrive::ISensorApi::TSensorsQueryOptions queryOptions;
                queryOptions.MutableFetchOptions().SetTimeout(SensorsTimeout);
                auto sensorsData = sensorApi->GetSensor(imeis, config->GetSensorId(), queryOptions, FetchReplication, MinSuccessfulReadsCount).ExtractValue(SensorsTimeout);
                Sensors = std::move(sensorsData);
            } catch (const yexception& e) {
                Context.AddError("TSensorIterator", e.what());
                return false;
            }
            return true;
        }

        bool ExtractData(TFetchedValue& data) const override {
            auto config = GetConfigAs<TSensorIteratorConfig>();
            if (!config) {
                Context.AddError("TSensorIterator", "could not get config");
                return false;
            }
            if (!config->GetSensorId()) {
                Context.AddError("TSensorIterator", "could not get sensor id");
                return false;
            }

            TString imei = Context.GetServer()->GetDriveAPI()->GetIMEI(GetCarId());
            if (!imei) {
                Context.AddError("TSensorIterator", "could not get imei for car id " + GetCarId());
                return false;
            }
            auto sensorsIt = Sensors.find(imei);
            if (sensorsIt == Sensors.end()) {
                Context.AddError("TSensorIterator", "could not get sensor for car id " + GetCarId() + " imei " + imei);
                return false;
            }
            for (auto&& sensorData : sensorsIt->second) {
                if (sensorData.Id == config->GetSensorId() && sensorData.GetSinceDelta(Context.GetFetchInstant()) >= config->GetMinSensorAge()) {
                    data = sensorData.ConvertTo<double>();
                    return true;
                }
            }
            return false;
        }

    private:
        NDrive::ISensorApi::TSensors Sensors;
        constexpr static ui32 FetchReplication = 4;
        constexpr static ui32 MinSuccessfulReadsCount = 2;
        constexpr static TDuration SensorsTimeout = TDuration::Seconds(20);
    };

    IFetchedIterator::TFactory::TRegistrator<TSensorIterator> TSensorIterator::Registrator(EFetchedItems::Sensor);
}
