#include "sensor_history.h"

#include <drive/backend/cars/car.h>

#include <drive/telematics/api/sensor/history.h>
#include <drive/telematics/server/sensors/cache.h>

bool TBaseSensorHistoryIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& value) {
    Y_UNUSED(value);
    return true;
}

NJson::TJsonValue TBaseSensorHistoryIteratorConfig::SerializeToJson() const {
    return {};
}

NDrive::TScheme TBaseSensorHistoryIteratorConfig::GetScheme(const IServerBase& server) const {
    Y_UNUSED(server);
    return {};
}

bool TBatteryDisconnectIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& value) {
    return
        TBase::DeserializeFromJson(value) &&
        NJson::ParseField(value["interval"], Interval) &&
        NJson::ParseField(value["connected_voltage_threshold"], ConnectedVoltageThreshold) &&
        NJson::ParseField(value["disconnected_voltage_threshold"], DisconnectedVoltageThreshold);
}

NJson::TJsonValue TBatteryDisconnectIteratorConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "interval", NJson::Hr(Interval));
    NJson::InsertField(result, "connected_voltage_threshold", ConnectedVoltageThreshold);
    NJson::InsertField(result, "disconnected_voltage_threshold", DisconnectedVoltageThreshold);
    return result;
}

NDrive::TScheme TBatteryDisconnectIteratorConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSDuration>("interval", "Интервал между последним и предыдущим значениями");
    result.Add<TFSNumeric>("connected_voltage_threshold", "Порог для предыдущего значения сенсора");
    result.Add<TFSNumeric>("disconnected_voltage_threshold", "Порог для последнего значения сенсора");
    return result;
}

NAlerts::TFetchedValue TBatteryDisconnectIterator::Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const {
    auto config = GetConfigAs<TBatteryDisconnectIteratorConfig>();
    Y_ENSURE(config);
    auto now = Now();
    auto current = sensorHistory.Get(VEGA_POWER_VOLTAGE, now);
    if (!current) {
        ERROR_LOG << "BatteryDisconnectIterator for " << objectId << ": no current voltage" << Endl;
        return 0;
    }
    auto previous = sensorHistory.Get(VEGA_POWER_VOLTAGE, now - config->GetInterval());
    if (!previous) {
        ERROR_LOG << "BatteryDisconnectIterator for " << objectId << ": no previous voltage" << Endl;
        return 0;
    }
    auto currentVoltage = current->ConvertTo<double>();
    auto previousVoltage = previous->ConvertTo<double>();
    NAlerts::TFetchedValue result =
        (currentVoltage <= config->GetDisconnectedVoltageThreshold() && previousVoltage > config->GetConnectedVoltageThreshold()) ? 1 : 0;
    NOTICE_LOG << "BatteryDisconnectIterator for " << objectId
        << ": current " << NJson::ToJson(current).GetStringRobust()
        << " previous " << NJson::ToJson(previous).GetStringRobust()
        << " result " << result << Endl;
    return result;
}

bool TEvacuationIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& value) {
    return
        TBase::DeserializeFromJson(value) &&
        NJson::ParseField(value["gps_movement_interval"], GpsMovementInterval) &&
        NJson::ParseField(value["prefilter_gps"], PrefilterGps) &&
        NJson::ParseField(value["prefilter_engine"], PrefilterEngine) &&
        NJson::ParseField(value["engine_off_interval"], EngineOffInterval);
}

NJson::TJsonValue TEvacuationIteratorConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "gps_movement_interval", NJson::Hr(GpsMovementInterval));
    NJson::InsertField(result, "engine_off_interval", NJson::Hr(EngineOffInterval));
    NJson::InsertField(result, "prefilter_gps", PrefilterGps);
    NJson::InsertField(result, "prefilter_engine", PrefilterEngine);
    return result;
}

NDrive::TScheme TEvacuationIteratorConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSDuration>("gps_movement_interval");
    result.Add<TFSDuration>("engine_off_interval");
    result.Add<TFSBoolean>("prefilter_gps");
    result.Add<TFSBoolean>("prefilter_engine");
    return result;
}

bool TBaseSensorHistoryIterator::ExtractData(NAlerts::TFetchedValue& data) const {
    auto config = GetConfigAs<TBaseSensorHistoryIteratorConfig>();
    if (!config) {
        return false;
    }
    const auto& objectId = GetCarId();
    try {
        const auto server = GetContext().GetServer();
        auto object = Yensured(server)->GetDriveDatabase().GetCarManager().GetObject(objectId);
        Y_ENSURE(object);
        if (!object->GetIMEI()) {
            INFO_LOG << "BaseSensorHistoryIterator for " << objectId << ": no imei" << Endl;
            return false;
        }

        auto preprocessed = Preprocess(objectId, *server);
        if (preprocessed) {
            data = *preprocessed;
            return true;
        }

        const auto client = Yensured(server)->GetSensorHistoryClient();
        const auto now = Now();
        const auto deadline = now + config->GetTimeout();
        const auto since = now - config->GetHistoryDepth();
        const auto until = TInstant::Max();
        const auto sensorIds = GetRequiredSensorIds(objectId);
        const auto asyncSensorHistory = Yensured(client)->Get(objectId, /*imei=*/{}, since, until, sensorIds);
        Y_ENSURE(asyncSensorHistory.Wait(deadline), "wait timeout for " << objectId);
        auto sensorHistory = asyncSensorHistory.GetValue();
        Y_ENSURE(sensorHistory, "no sensor history for " << objectId);
        data = Process(objectId, *sensorHistory);
        return true;
    } catch (...) {
        ERROR_LOG << "BaseSensorHistoryIterator for " << objectId << ": " << CurrentExceptionInfo().GetStringRobust() << Endl;
        return false;
    }
}

TVector<NDrive::TSensorId> TBaseSensorHistoryIterator::GetRequiredSensorIds(const TString& /*objectId*/) const {
    return {};
}

TMaybe<NAlerts::TFetchedValue> TBaseSensorHistoryIterator::Preprocess(const TString& /*objectId*/, const NDrive::IServer& /*server*/) const {
    return {};
}

TVector<NDrive::TSensorId> TEvacuationIterator::GetRequiredSensorIds(const TString& /*objectId*/) const {
    return {
        CAN_ENGINE_IS_ON,
        VEGA_SPEED,
    };
}

bool TEvacuationIterator::InitByObjects(IFetchedIterator& objectIterator) {
    TSet<TString> objectIds;
    for (; !objectIterator.IsFinished(); objectIterator.Next()) {
        auto objectId = objectIterator.GetObjectId(NAlerts::EAlertEntityType::Car);
        objectIds.insert(ToString(objectId));
    }
    auto server = Context.GetServer();
    DeviceSnapshots = Yensured(server)->GetSnapshotsManager().GetSnapshots(objectIds);
    return true;
}

TMaybe<NAlerts::TFetchedValue> TEvacuationIterator::Preprocess(const TString& objectId, const NDrive::IServer& server) const {
    Y_UNUSED(server);
    auto config = GetConfigAs<TEvacuationIteratorConfig>();
    Y_ENSURE(config);
    if (!config->GetPrefilterGps() && !config->GetPrefilterEngine()) {
        return {};
    }

    auto snapshot = DeviceSnapshots.GetSnapshot(objectId);
    if (!snapshot) {
        ERROR_LOG << "EvacuationIterator for " << objectId << ": no snapshot" << Endl;
        return 0;
    }
    if (config->GetPrefilterGps()) {
        auto speed = snapshot->GetSensor(VEGA_SPEED, config->GetGpsMovementInterval());
        if (speed) {
            auto value = speed->ConvertTo<double>();
            if (value < 0.001 && speed->Since <= Now() - config->GetGpsMovementInterval()) {
                DEBUG_LOG << "EvacuationIterator for " << objectId << ": preprocessed gps on " << NJson::ToJson(speed).GetStringRobust() << Endl;
                return 0;
            }
        }
    }
    if (config->GetPrefilterEngine()) {
        auto engineOn = snapshot->GetSensor(CAN_ENGINE_IS_ON, config->GetEngineOffInterval());
        if (engineOn && engineOn->ConvertTo<bool>()) {
            DEBUG_LOG << "EvacuationIterator for " << objectId << ": preprocessed engine on " << NJson::ToJson(engineOn).GetStringRobust() << Endl;
            return 0;
        }
    }
    return {};
}

NAlerts::TFetchedValue TEvacuationIterator::Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const {
    auto config = GetConfigAs<TEvacuationIteratorConfig>();
    Y_ENSURE(config);
    bool debug = config->IsDebug();
    {
        auto now = Now();
        auto speedSince = now - config->GetGpsMovementInterval();
        auto speedHistory = sensorHistory.GetRange(VEGA_SPEED, speedSince, now);
        ui32 moving = 0;
        for (auto&& sensor : speedHistory) {
            auto value = sensor.ConvertTo<double>();
            if (value >= config->GetGpsMovementSpeedThreshold()) {
                ++moving;
            }
        }
        if (moving < config->GetGpsMovementCountThreshold()) {
            if (debug) {
                INFO_LOG << "EvacuationIterator for " << objectId << ": not moving " << NJson::ToJson(speedHistory).GetStringRobust() << Endl;
            }
            return 0;
        }

        auto engineOnSince = now - config->GetEngineOffInterval();
        auto engineOnSensor = sensorHistory.Get(CAN_ENGINE_IS_ON, engineOnSince);
        Y_ENSURE(engineOnSensor);
        auto engineOn = engineOnSensor->ConvertTo<bool>();
        if (engineOn) {
            if (debug) {
                INFO_LOG << "EvacuationIterator for " << objectId << ": engine on " << NJson::ToJson(engineOnSensor).GetStringRobust() << Endl;
            }
            return 0;
        }
        auto engineOnHistory = sensorHistory.GetRange(CAN_ENGINE_IS_ON, engineOnSince, now);
        for (auto&& sensor : engineOnHistory) {
            if (sensor.ConvertTo<bool>()) {
                if (debug) {
                    INFO_LOG << "EvacuationIterator for " << objectId << ": engine on " << NJson::ToJson(sensor).GetStringRobust() << Endl;
                }
                return 0;
            }
        }
        NOTICE_LOG << "EvacuationIterator positive for " << objectId << ": " << NJson::ToJson(speedHistory).GetStringRobust() << "," << NJson::ToJson(engineOnHistory).GetStringRobust() << Endl;
    }
    return 1;
}

bool TFuelChangeIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& value) {
    return
        TBase::DeserializeFromJson(value) &&
        NJson::ParseField(value["threshold"], Threshold);
}

NJson::TJsonValue TFuelChangeIteratorConfig::SerializeToJson() const {
    NJson::TJsonValue result = TBase::SerializeToJson();
    NJson::InsertField(result, "threshold", Threshold);
    return result;
}

NDrive::TScheme TFuelChangeIteratorConfig::GetScheme(const IServerBase& server) const {
    NDrive::TScheme result = TBase::GetScheme(server);
    result.Add<TFSNumeric>("threshold");
    return result;
}

NAlerts::TFetchedValue TFuelChangeIterator::Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const {
    auto config = GetConfigAs<TFuelChangeIteratorConfig>();
    Y_ENSURE(config);
    auto fuelLevelValues = sensorHistory.GetRange(CAN_FUEL_LEVEL_P);
    auto fuelLevelValuesCount = fuelLevelValues.size();
    if (fuelLevelValuesCount < 2) {
        ERROR_LOG << "FuelChangeIterator for " << objectId << ": values count " << fuelLevelValuesCount << Endl;
        return 0;
    }

    auto now = Now();
    auto since = now - config->GetInterval();

    TMaybe<NDrive::TSensor> previous;
    for (auto&& sensor : Reversed(fuelLevelValues)) {
        if (previous) {
            auto delta = previous->ConvertTo<double>() - sensor.ConvertTo<double>();
            bool positive = config->GetPositive();
            if (
                (positive && delta > config->GetThreshold()) ||
                (!positive && delta < -1 * config->GetThreshold())
            ) {
                NOTICE_LOG << "FuelChangeIterator for " << objectId << ": from " << NJson::ToJson(sensor).GetStringRobust() << " to " << NJson::ToJson(previous).GetStringRobust() << Endl;
                return 1;
            }
        }
        previous = sensor;
        if (sensor.Timestamp < since) {
            break;
        }
    }

    return 0;
}

NAlerts::IIteratorConfig::TFactory::TRegistrator<TBatteryDisconnectIteratorConfig> TBatteryDisconnectIteratorConfig::Registrator(NAlerts::EFetchedItems::BatteryDisconnect);
NAlerts::IFetchedIterator::TFactory::TRegistrator<TBatteryDisconnectIterator> TBatteryDisconnectIterator::Registrator(NAlerts::EFetchedItems::BatteryDisconnect);

NAlerts::IIteratorConfig::TFactory::TRegistrator<TEvacuationIteratorConfig> TEvacuationIteratorConfig::Registrator(NAlerts::EFetchedItems::Evacuation);
NAlerts::IFetchedIterator::TFactory::TRegistrator<TEvacuationIterator> TEvacuationIterator::Registrator(NAlerts::EFetchedItems::Evacuation);

NAlerts::IIteratorConfig::TFactory::TRegistrator<TFuelChangeIteratorConfig> TFuelChangeIteratorConfig::Registrator(NAlerts::EFetchedItems::FuelChange);
NAlerts::IFetchedIterator::TFactory::TRegistrator<TFuelChangeIterator> TFuelChangeIterator::Registrator(NAlerts::EFetchedItems::FuelChange);
