#include "iterators.h"

#include <drive/backend/device_snapshot/manager.h>

namespace NDrive {
    class TSensorsCache;
}

class TBaseSensorHistoryIteratorConfig: public NAlerts::IIteratorConfig {
private:
    using TBase = IIteratorConfig;

public:
    R_FIELD(TDuration, HistoryDepth, TDuration::Days(1));
    R_FIELD(TDuration, Timeout, TDuration::Seconds(1));

public:
    using TBase::TBase;

    bool DeserializeFromJson(const NJson::TJsonValue& json) override;
    NJson::TJsonValue SerializeToJson() const override;
    NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
};

class TBaseSensorHistoryIterator: public NAlerts::TContainerIteratorBase {
private:
    using TBase = TContainerIteratorBase;

public:
    using TBase::TBase;

    TString GetCarId() const {
        auto session = GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>();
        if (!session) {
            return TString(GetObjectId().Data(), GetObjectId().Size());
        }
        return session->GetObjectId();
    }

    bool ExtractData(NAlerts::TFetchedValue& data) const override;

protected:
    virtual TVector<NDrive::TSensorId> GetRequiredSensorIds(const TString& objectId) const;
    virtual TMaybe<NAlerts::TFetchedValue> Preprocess(const TString& objectId, const NDrive::IServer& server) const;
    virtual NAlerts::TFetchedValue Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const = 0;
};

class TBatteryDisconnectIteratorConfig: public TBaseSensorHistoryIteratorConfig {
private:
    using TBase = TBaseSensorHistoryIteratorConfig;

public:
    R_FIELD(TDuration, Interval, TDuration::Minutes(5));
    R_FIELD(double, ConnectedVoltageThreshold, 9);
    R_FIELD(double, DisconnectedVoltageThreshold, 5);

public:
    using TBase::TBase;

    bool DeserializeFromJson(const NJson::TJsonValue& json) override;
    NJson::TJsonValue SerializeToJson() const override;
    NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;

private:
    static NAlerts::IIteratorConfig::TFactory::TRegistrator<TBatteryDisconnectIteratorConfig> Registrator;
};

class TBatteryDisconnectIterator: public TBaseSensorHistoryIterator {
private:
    using TBase = TBaseSensorHistoryIterator;

public:
    using TBase::TBase;

    NAlerts::EFetchedItems GetField() const override {
        return NAlerts::EFetchedItems::BatteryDisconnect;
    }

protected:
    NAlerts::TFetchedValue Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const override;

private:
    static NAlerts::IFetchedIterator::TFactory::TRegistrator<TBatteryDisconnectIterator> Registrator;
};

class TEvacuationIteratorConfig: public TBaseSensorHistoryIteratorConfig {
private:
    using TBase = TBaseSensorHistoryIteratorConfig;

public:
    R_FIELD(TDuration, EngineOffInterval, TDuration::Minutes(10));
    R_FIELD(TDuration, GpsMovementInterval, TDuration::Minutes(5));
    R_FIELD(double, GpsMovementSpeedThreshold, 10);
    R_FIELD(ui32, GpsMovementCountThreshold, 5);
    R_FIELD(bool, PrefilterEngine, true);
    R_FIELD(bool, PrefilterGps, true);
    R_FIELD(bool, Debug, false);

public:
    using TBase::TBase;

    bool DeserializeFromJson(const NJson::TJsonValue& json) override;
    NJson::TJsonValue SerializeToJson() const override;
    NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;

private:
    static NAlerts::IIteratorConfig::TFactory::TRegistrator<TEvacuationIteratorConfig> Registrator;
};

class TEvacuationIterator: public TBaseSensorHistoryIterator {
private:
    using TBase = TBaseSensorHistoryIterator;

public:
    using TBase::TBase;

    NAlerts::EFetchedItems GetField() const override {
        return NAlerts::EFetchedItems::Evacuation;
    }

    bool InitByObjects(IFetchedIterator& objectIterator) override;

protected:
    TVector<NDrive::TSensorId> GetRequiredSensorIds(const TString& objectId) const override;
    TMaybe<NAlerts::TFetchedValue> Preprocess(const TString& objectId, const NDrive::IServer& server) const override;
    NAlerts::TFetchedValue Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const override;

private:
    TDevicesSnapshot DeviceSnapshots;

private:
    static NAlerts::IFetchedIterator::TFactory::TRegistrator<TEvacuationIterator> Registrator;
};

class TFuelChangeIteratorConfig: public TBaseSensorHistoryIteratorConfig {
private:
    using TBase = TBaseSensorHistoryIteratorConfig;

public:
    R_FIELD(TDuration, Interval, TDuration::Minutes(5));
    R_FIELD(double, Threshold, 30);
    R_FIELD(bool, Positive, false);

public:
    using TBase::TBase;

    bool DeserializeFromJson(const NJson::TJsonValue& json) override;
    NJson::TJsonValue SerializeToJson() const override;
    NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;

private:
    static NAlerts::IIteratorConfig::TFactory::TRegistrator<TFuelChangeIteratorConfig> Registrator;
};

class TFuelChangeIterator: public TBaseSensorHistoryIterator {
private:
    using TBase = TBaseSensorHistoryIterator;

public:
    using TBase::TBase;

    NAlerts::EFetchedItems GetField() const override {
        return NAlerts::EFetchedItems::Evacuation;
    }

protected:
    NAlerts::TFetchedValue Process(const TString& objectId, const NDrive::TSensorsCache& sensorHistory) const override;

private:
    static NAlerts::IFetchedIterator::TFactory::TRegistrator<TFuelChangeIterator> Registrator;
};
