#include "iterators.h"

#include <drive/backend/data/chargable.h>
#include <drive/backend/offers/actions/pack.h>

namespace NAlerts {
    class TSessionStateTimeIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TSessionStateTimeIterator> Registrator;

    public:
        using TBase::TBase;

        NAlerts::EFetchedItems GetField() const override {
            return NAlerts::EFetchedItems::SessionStateTime;
        }

        bool ExtractData(NAlerts::TFetchedValue& data) const override {
            auto session = GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>();
            if (!session) {
                return false;
            }
            auto lastEvent = session->GetLastEvent();
            CHECK_WITH_LOG(lastEvent.Defined());
            data = (Context.GetFetchInstant() - lastEvent->GetHistoryInstant()).Seconds();
            return true;
        }
    };

    class TSessionBillIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TSessionBillIterator> Registrator;

    public:
        using TBase::TBase;

        NAlerts::EFetchedItems GetField() const override {
            return NAlerts::EFetchedItems::SessionBill;
        }

        bool ExtractData(NAlerts::TFetchedValue& data) const override {
            auto session = GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>();
            if (!session) {
                return false;
            }
            TBillingSession::TBillingCompilation compilation;
            if (!session->FillCompilation(compilation)) {
                return false;
            }
            data = compilation.GetBillingSumPrice();
            return true;
        }
    };

    TMaybe<TOfferStatePtr> GetOfferState(const IEventsSession<TCarTagHistoryEvent>* session) {
        if (!session) {
            return Nothing();
        }
        TBillingSession::TBillingCompilation compilation;
        if (!session->FillCompilation(compilation)) {
            return Nothing();
        }
        auto state = compilation.GetCurrentOfferState();
        if (!state) {
            return Nothing();
        }
        return state;
    }

    class TPackOfferRemainingDuration : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TPackOfferRemainingDuration> Registrator;

    public:
        using TBase::TBase;

        NAlerts::EFetchedItems GetField() const override {
            return NAlerts::EFetchedItems::PackOfferRemainingDuration;
        }

        bool ExtractData(NAlerts::TFetchedValue& data) const override {
            auto session = GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>();
            auto offerState = GetOfferState(session);
            if (!offerState) {
                return false;
            }
            auto packOfferState = dynamic_cast<TPackOfferState*>((*offerState).Get());
            if (packOfferState) {
                data = packOfferState->GetRemainingTime().Seconds();
                return true;
            }
            return false;
        }
    };

    class TPackOfferRemainingDistance : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TPackOfferRemainingDistance> Registrator;

    public:
        using TBase::TBase;

        NAlerts::EFetchedItems GetField() const override {
            return NAlerts::EFetchedItems::PackOfferRemainingDistance;
        }

        bool ExtractData(NAlerts::TFetchedValue& data) const override {
            auto session = GetDataCurrent<::IEventsSession<TCarTagHistoryEvent>>();
            auto offerState = GetOfferState(session);
            if (!offerState) {
                return false;
            }
            auto packOfferState = dynamic_cast<TPackOfferState*>((*offerState).Get());
            if (packOfferState) {
                data = packOfferState->GetRemainingDistance() * 1000.0;
                return true;
            }
            return false;
        }
    };

    class TActualSessionsIterator : public TContainerIteratorBase {
    private:
        using TBase = TContainerIteratorBase;
        using TEventsSessionPtr = IEventsSession<TObjectEvent<TConstDBTag>>::TPtr;

    public:
        using TBase::TBase;

        bool InitByObjects(IFetchedIterator& iterator) override {
            auto builder = Context.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", Context.GetFetchInstant());
            if (!builder) {
                return false;
            }
            switch (iterator.GetEntityType()) {
            case EAlertEntityType::Car:
                ActualSessions = builder->GetSessionsActualByObjects();
                break;
            case EAlertEntityType::User: {
                auto sessions = builder->GetSessionsActual();
                for (auto&& session : sessions) {
                    ActualSessions.emplace(session->GetUserId(), session);
                }
                break;
            }
            case EAlertEntityType::Session: {
                auto sessions = builder->GetSessionsActual();
                for (auto&& session : sessions) {
                    ActualSessions.emplace(session->GetSessionId(), session);
                }
                break;
            }
            default:
                return false;
            }
            return true;
        }

        EFetchedItems GetField() const override {
            return EFetchedItems::ActualSessionsCount;
        }

        bool ExtractData(TFetchedValue& data) const override {
            TString object(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
            data = ActualSessions.count(object);
            return true;
        }

    private:
        std::multimap<TString, TEventsSessionPtr> ActualSessions;
        static IFetchedIterator::TFactory::TRegistrator<TActualSessionsIterator> Registrator;
    };

    IFetchedIterator::TFactory::TRegistrator<TSessionStateTimeIterator> TSessionStateTimeIterator::Registrator(EFetchedItems::SessionStateTime);
    IFetchedIterator::TFactory::TRegistrator<TSessionBillIterator> TSessionBillIterator::Registrator(EFetchedItems::SessionBill);
    IFetchedIterator::TFactory::TRegistrator<TPackOfferRemainingDuration> TPackOfferRemainingDuration::Registrator(EFetchedItems::PackOfferRemainingDuration);
    IFetchedIterator::TFactory::TRegistrator<TPackOfferRemainingDistance> TPackOfferRemainingDistance::Registrator(EFetchedItems::PackOfferRemainingDistance);
    IFetchedIterator::TFactory::TRegistrator<TActualSessionsIterator> TActualSessionsIterator::Registrator(EFetchedItems::ActualSessionsCount);
}
