#include "tags.h"

#include <drive/backend/data/scoring/scoring.h>

namespace NAlerts {
    IFetchedIterator::TFactory::TRegistrator<TTagFieldCountIterator> TTagFieldCountIterator::Registrator(EFetchedItems::TagField);
    IFetchedIterator::TFactory::TRegistrator<TScoringTagIterator> TScoringTagIterator::Registrator(EFetchedItems::ScoringTag);
    IFetchedIterator::TFactory::TRegistrator<TTagFieldValueIterator> TTagFieldValueIterator::Registrator(EFetchedItems::TagFieldValue);
    IFetchedIterator::TFactory::TRegistrator<TSLADeadlineIterator> TSLADeadlineIterator::Registrator(EFetchedItems::SLADeadline);

    bool ITagFieldIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "tag_name", TagName, true) &&
            NJson::ParseField(json, "field_path", FieldPath, true);
    }

    NJson::TJsonValue ITagFieldIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "tag_name", TagName);
        NJson::InsertField(result, "field_path", FieldPath);
        return result;
    }

    NDrive::TScheme ITagFieldIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("tag_name", "Имя тега").SetRequired(true);
        scheme.Add<TFSString>("field_path", "Путь до поля через '.'").SetRequired(true);
        return scheme;
    }


    bool TTagFieldCountIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            TBase::DeserializeFromJson(json) &&
            NJson::ParseField(json, "field_values", FieldValues, true);
    }

    NJson::TJsonValue TTagFieldCountIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result = TBase::SerializeToJson();
        NJson::InsertField(result, "field_values", FieldValues);
        return result;
    }

    NDrive::TScheme TTagFieldCountIteratorConfig::GetScheme(const IServerBase& server) const {
        NDrive::TScheme scheme = TBase::GetScheme(server);
        scheme.Add<TFSArray>("field_values").SetElement<TFSString>();
        return scheme;
    }

    bool ITagFieldIterator::InitByObjects(IFetchedIterator& objectIterator) {
        const TFetcherContext& context = TBase::Context;
        TSet<TString> objects;
        while (!objectIterator.IsFinished()) {
            auto object = objectIterator.GetObjectId();
            objects.emplace(object.Data(), object.Size());
            objectIterator.Next();
        }
        if (objects.empty()) {
            return true;
        }
        auto config = TBase::template GetConfigAs<ITagFieldIteratorConfig>();
        if (!config) {
            return false;
        }
        auto tagEntity = NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
        if (!tagEntity.Defined()) {
            context.AddError("TEventTagEventCountIterator", TStringBuilder() <<
                                "Bad entity type " << ToString(GetEntityType()) << " for TTagFieldIterator of fetcher " << ToString(GetFetcherName()));
            return false;
        }
        const auto& tagsManager = context.GetServer()->GetDriveAPI()->GetEntityTagsManager(tagEntity.GetRef());
        auto session = tagsManager.BuildSession(true);
        auto tags = tagsManager.RestoreTags(objects, { config->GetTagName() }, session);
        if (!tags) {
            context.AddError("TEventTagEventCountIterator", TStringBuilder() <<
                                "Could not restore tags in TTagFieldIterator of fetcher " << ToString(GetFetcherName()));
            return false;
        }
        for (auto&& tag : *tags) {
            ObjectsTags[tag.GetObjectId()].emplace_back(std::move(tag));
        }
        return true;
    }

    bool ITagFieldIterator::ExtractData(TFetchedValue& data) const {
        TString objectId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        return ValueFromTags(ObjectsTags.Value(objectId, TVector<TDBTag>()), data);
    }

    bool TTagFieldCountIterator::ValueFromTags(const TVector<TDBTag>& tags, TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TTagFieldCountIteratorConfig>();
        if (!config) {
            return false;
        }
        if (tags.empty()) {
            return true;
        }
        data = 0;
        for (auto&& tag : tags) {
            auto json = tag.BuildJsonReport();
            NJson::TJsonValue value;
            if (json.GetValueByPath(config->GetFieldPath(), value)) {
                const TString& foundValue = value.GetStringRobust();
                for (auto&& fieldValue : config->GetFieldValues()) {
                    if (foundValue == fieldValue) {
                        ++data;
                    }
                }
            } else {
                ERROR_LOG << "Can't get tag " << config->GetTagName() << " value by path " << config->GetFieldPath() << " in TTagFieldIterator of fetcher " << ToString(GetFetcherName()) << Endl;
            }
        }
        return true;
    }

    bool TScoringTagIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "tag_name", TagName, true) &&
            NJson::ParseField(json, "scoring_min", ScoringMin, true) &&
            NJson::ParseField(json, "scoring_max", ScoringMax, true);
    }

    NJson::TJsonValue TScoringTagIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "tag_name", TagName);
        NJson::InsertField(result, "scoring_min", ScoringMin);
        NJson::InsertField(result, "scoring_max", ScoringMax);
        return result;
    }

    NDrive::TScheme TScoringTagIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("tag_name", "Имя тега").SetRequired(true);
        scheme.Add<TFSNumeric>("scoring_min", "Минимальное значение");
        scheme.Add<TFSNumeric>("scoring_max", "Максимальное значение");
        return scheme;
    }

    TScoringTagIterator::TScoringTagIterator(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig)
        : TBase(fetcherType, entityType, context, iteratorConfig)
    {
        auto config = TBase::template GetConfigAs<TScoringTagIteratorConfig>();
        Y_ENSURE_BT(config);
        if (!config->GetTagName()) {
            return;
        }
        const auto& tagsManager = context.GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags();
        auto session = tagsManager.BuildSession(true);
        auto tags = tagsManager.RestoreTags(TSet<TString>(), { config->GetTagName() }, session);
        if (!tags) {
            ERROR_LOG << "TScoringTagIterator could not restore tags " << session.GetStringReport() << Endl;
            return;
        }
        for (auto&& tag : *tags) {
            const TScoringUserTag* ScoringTag = tag.GetTagAs<TScoringUserTag>();
            if (!ScoringTag) {
                ERROR_LOG << "TScoringTagIterator tag is not scoring tag " << session.GetStringReport() << Endl;
                return;
            }
            UserScorings[tag.GetObjectId()].emplace_back(ScoringTag->GetValue());
        }
    }

    bool TScoringTagIterator::ExtractData(TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TScoringTagIteratorConfig>();
        if (!config) {
            return false;
        }

        TString objectId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        auto it = UserScorings.find(objectId);
        if (it == UserScorings.end()) {
            return false;
        }
        data = 0;
        for (auto&& scoring : it->second) {
            if (config->GetScoringMin() <= scoring && scoring < config->GetScoringMax()) {
                data = 1;
                return true;
            }
        }
        return true;
    }

    bool TTagFieldValueIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            TBase::DeserializeFromJson(json) &&
            NJson::ParseField(json, "precision", Precision, false) &&
            NJson::ParseField(json, "default_value", DefaultValue);
    }

    NJson::TJsonValue TTagFieldValueIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result = TBase::SerializeToJson();
        NJson::InsertField(result, "precision", Precision);
        NJson::InsertNonNull(result, "default_value", DefaultValue);
        return result;
    }

    NDrive::TScheme TTagFieldValueIteratorConfig::GetScheme(const IServerBase& server) const {
        NDrive::TScheme scheme = TBase::GetScheme(server);
        scheme.Add<TFSNumeric>("default_value");
        scheme.Add<TFSNumeric>("precision");
        return scheme;
    }

    bool TTagFieldValueIterator::ValueFromTags(const TVector<TDBTag>& tags, TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TTagFieldValueIteratorConfig>();
        if (!config) {
            return false;
        }
        data = 0;
        for (auto&& tag : tags) {
            auto json = tag.BuildJsonReport();
            NJson::TJsonValue value;
            if (json.GetValueByPath(config->GetFieldPath(), value)) {
                double origValue = value.GetDoubleRobust();
                origValue *= std::pow(10, config->GetPrecision());
                data = std::round(origValue);
                return true;
            }
        }
        if (config->HasDefaultValue()) {
            data = config->GetDefaultValueRef();
            return true;
        }
        return false;
    }

    bool TSLADeadlineIteratorConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
        return
            NJson::ParseField(json, "tag_name", TagName, true);
    }

    NJson::TJsonValue TSLADeadlineIteratorConfig::SerializeToJson() const {
        NJson::TJsonValue result;
        NJson::InsertField(result, "tag_name", TagName);
        return result;
    }

    NDrive::TScheme TSLADeadlineIteratorConfig::GetScheme(const IServerBase& /*server*/) const {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("tag_name", "Имя тега").SetRequired(true);
        return scheme;
    }

    bool TSLADeadlineIterator::InitByObjects(IFetchedIterator& iterator) {
        auto config = TBase::template GetConfigAs<TSLADeadlineIteratorConfig>();
        if (iterator.GetEntityType() !=  EAlertEntityType::User || !config->GetTagName()) {
            return false;
        }
        const auto& tagsManager = Context.GetServer()->GetDriveAPI()->GetTagsManager().GetUserTags();
        auto session = tagsManager.BuildTx<NSQL::ReadOnly>();
        auto tags = tagsManager.RestoreTags(TSet<TString>(), { config->GetTagName() }, session);
        if (!tags) {
            ERROR_LOG << "TSLADeadlineIterator could not restore tags " << session.GetStringReport() << Endl;
            return false;
        }
        for (auto&& tag : *tags) {
            auto tagData = tag.GetData();
            if (!tagData) {
                ERROR_LOG << "TSLADeadlineIterator ITag::TConstPtr is null in tag" << session.GetStringReport() << Endl;
                return false;
            }
            TagsSLA[tag.GetObjectId()] = tagData->GetSLAInstant();
        }
        return true;
    }

    bool TSLADeadlineIterator::ExtractData(TFetchedValue& data) const {
        auto config = TBase::template GetConfigAs<TSLADeadlineIteratorConfig>();
        if (!config) {
            return false;
        }

        TString objectId(TBase::GetObjectId().Data(), TBase::GetObjectId().Size());
        auto it = TagsSLA.find(objectId);
        if (it == TagsSLA.end()) {
            return true;
        }
        auto&& sla = it->second;
        if (sla != TInstant::Zero()) {
            data = (sla - ModelingNow()).Seconds();
            return true;
        }
        return true;
    }
}
