#pragma once
#include "iterators.h"

#include <drive/backend/data/event_tag.h>

namespace NAlerts {
    class ITagFieldIteratorConfig: public IIteratorConfig {
    private:
        using TBase = IIteratorConfig;
        R_READONLY(TString, TagName);
        R_READONLY(TString, FieldPath);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TTagFieldCountIteratorConfig: public ITagFieldIteratorConfig {
    private:
        using TBase = ITagFieldIteratorConfig;
        R_READONLY(TVector<TString>, FieldValues);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class ITagFieldIterator: public TContainerIteratorBase {
    private:
        using TBase = TContainerIteratorBase;

    public:
        using TBase::TBase;

        virtual bool InitByObjects(IFetchedIterator& objectIterator) override final;
        virtual bool ExtractData(TFetchedValue& data) const override final;

    private:
        virtual bool ValueFromTags(const TVector<TDBTag>& tags, TFetchedValue& data) const = 0;

    protected:
        TMap<TString, TVector<TDBTag>> ObjectsTags;
    };

    class TTagFieldCountIterator: public ITagFieldIterator {
    private:
        using TBase = ITagFieldIterator;
        static IFetchedIterator::TFactory::TRegistrator<TTagFieldCountIterator> Registrator;

    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::TagField;
        }

        virtual bool ValueFromTags(const TVector<TDBTag>& tags, TFetchedValue& data) const override;
    };

    class TScoringTagIteratorConfig: public IIteratorConfig {
    private:
        using TBase = IIteratorConfig;
        R_READONLY(TString, TagName);
        R_READONLY(double, ScoringMin, 0);
        R_READONLY(double, ScoringMax, 0);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TScoringTagIterator: public TContainerIteratorBase {
    private:
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TScoringTagIterator> Registrator;

    public:
        using TBase::TBase;

        TScoringTagIterator(const EDataFetcherType fetcherType, const EAlertEntityType entityType, const TFetcherContext& context, const IIteratorConfig::TPtr& iteratorConfig);

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::ScoringTag;
        }

        bool ExtractData(TFetchedValue& data) const override;

    private:
        TMap<TString, TVector<double>> UserScorings;
    };

    class TTagFieldValueIteratorConfig: public ITagFieldIteratorConfig {
    private:
        using TBase = ITagFieldIteratorConfig;
        R_OPTIONAL(ui64, DefaultValue);
        R_READONLY(ui8, Precision, 0);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TTagFieldValueIterator: public ITagFieldIterator {
    private:
        using TBase = ITagFieldIterator;
        static IFetchedIterator::TFactory::TRegistrator<TTagFieldValueIterator> Registrator;

    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::TagFieldValue;
        }

        virtual bool ValueFromTags(const TVector<TDBTag>& tags, TFetchedValue& data) const override;
    };

    class TSLADeadlineIteratorConfig: public IIteratorConfig {
    private:
        using TBase = IIteratorConfig;
        R_READONLY(TString, TagName);

    public:
        using TBase::TBase;
        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const override;
    };

    class TSLADeadlineIterator: public TContainerIteratorBase {
    private:
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TSLADeadlineIterator> Registrator;

    public:
        using TBase::TBase;

        bool InitByObjects(IFetchedIterator& iterator) override;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::SLADeadline;
        }

        bool ExtractData(TFetchedValue& data) const override;

    private:
        TMap<TString, TInstant> TagsSLA;
    };
}
