#pragma once

#include "iterators.h"

#include <drive/backend/data/dictionary_tags.h>

namespace NAlerts {

    class TRegistrationTimeIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TRegistrationTimeIterator> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::RegApprovedDelta;
        }

        virtual bool ExtractData(TFetchedValue& data) const override {
            auto currentData = GetDataCurrent<TDriveUserData>();
            if (!currentData) {
                return false;
            }
            data = (Context.GetFetchInstant() - currentData->GetApprovedAt()).Seconds();
            return true;
        }
    };

    class TIsPhoneVerifiedIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TIsPhoneVerifiedIterator> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::IsPhoneVerified;
        }

        virtual bool ExtractData(TFetchedValue& data) const override {
            auto currentData = GetDataCurrent<TDriveUserData>();
            if (!currentData) {
                return false;
            }
            data = currentData->IsPhoneVerified() ? 1 : 0;
            return true;
        }
    };

    class TIsMailVerifiedIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TIsMailVerifiedIterator> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::IsMailVerified;
        }

        virtual bool ExtractData(TFetchedValue& data) const override {
            const TFetcherContext& context = TBase::Context;

            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TDBTag settingsTag = context.GetServer()->GetDriveAPI()->GetUserSettings(TString(TBase::GetObjectId().Data(), TBase::GetObjectId().Size()), session);
            TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
            if (!settingsData) {
                return false;
            }
            auto emailStatus = settingsData->GetField("verified_mail");
            if (emailStatus.Defined()) {
                data = IsTrue(emailStatus.GetRef());
            }
            return true;
        }
    };

    class TBindMailTimeIterator : public TContainerIteratorBase {
        using TBase = TContainerIteratorBase;
        static IFetchedIterator::TFactory::TRegistrator<TBindMailTimeIterator> Registrator;
    public:
        using TBase::TBase;

        virtual EFetchedItems GetField() const override {
            return EFetchedItems::BindMailTime;
        }

        virtual bool ExtractData(TFetchedValue& data) const override {
            const TFetcherContext& context = TBase::Context;

            auto session = context.GetServer()->GetDriveAPI()->template BuildTx<NSQL::ReadOnly>();
            TDBTag settingsTag = context.GetServer()->GetDriveAPI()->GetUserSettings(TString(TBase::GetObjectId().Data(), TBase::GetObjectId().Size()), session);
            TUserDictionaryTag* settingsData = settingsTag.MutableTagAs<TUserDictionaryTag>();
            if (!settingsData) {
                return false;
            }

            auto emailTime = settingsData->GetField("sending_verification_mail_time");
            if (emailTime.Defined()) {
                ui64 emailTimeSeconds = 0;
                if (TryFromString(emailTime.GetRef(), emailTimeSeconds)) {
                    data = (context.GetFetchInstant() - TInstant::Seconds(emailTimeSeconds)).Seconds();
                } else {
                    return false;
                }
            }
            return true;
        }
    };
}
