#include "condition.h"

#include <drive/backend/alerts/data/iterators.h>

#include <util/string/vector.h>

using namespace NAlerts;

void TFetcherContext::AddError(const TString& location, const TString& error) const {
    Errors.AddMessage(location, error);
}

bool TChainContext::DeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING_OPT(json, "state_tag", StateTagName);
    JREAD_DURATION_OPT(json, "state_interval", StateInterval);
    if (!TJsonProcessor::ReadContainer(json, "acceptable_states", AcceptableStates)) {
        return false;
    }
    return true;
}

void TChainContext::SerializeToJson(NJson::TJsonValue& result) const {
    JWRITE(result, "state_tag", StateTagName);
    JWRITE_DURATION(result, "state_interval", StateInterval);
    TJsonProcessor::WriteContainerArray(result, "acceptable_states", AcceptableStates);
}

void TChainContext::GetScheme(NDrive::TScheme& scheme, const TVector<TString>& alertTags) {
    scheme.Add<TFSVariants>("state_tag", "Тег алерта").SetVariants(alertTags);
    scheme.Add<TFSArray>("acceptable_states", "Выполнять только из состояний").SetElement<TFSString>();
    scheme.Add<TFSDuration>("state_interval", "Минимальный интервал после предыдущего состояния").SetDefault(TDuration::Zero());
}

bool TChainContext::CheckState(const TVector<TString>& statesHistory, const TVector<TInstant>& timeline, const TInstant checkInstant) const {
    TString prevState = statesHistory.empty() ? "none" : statesHistory.back();

    if (AcceptableStates.contains(prevState) || (AcceptableStates.empty() && (prevState == "none"))) {
        if (!timeline.empty() && checkInstant < timeline.back() + StateInterval) {
            return false;
        }
        return true;
    }
    return false;
}


NDrive::TScheme IServiceDataFetcherConfig::GetScheme(const IServerBase& server)  const {
    auto scheme = DoGetScheme(server);
    scheme.Add<TFSVariants>("type", "Тип фетчера").InitVariantsClass<IServiceDataFetcherConfig, NAlerts::EDataFetcherType>();
    scheme.Add<TFSNumeric>("index", "Номер фетчера по порядку выполнения").SetDefault(0);
    return scheme;
}

bool IServiceDataFetcherConfig::DeserializeFromJson(const NJson::TJsonValue& json) {
    if (!NJson::ParseField(json["index"], Index)) {
        return false;
    }
    return DoDeserializeFromJson(json);
}

NJson::TJsonValue IServiceDataFetcherConfig::SerializeToJson() const {
    auto json = DoSerializeToJson();
    NJson::InsertField(json, "type", ToString(GetFetcherType()));
    NJson::InsertField(json, "index", Index);
    return json;
}

TSet<EAlertEntityType> IServiceDataFetcherConfig::GetAcceptableEntityTypes() const {
    return { GetEntityType() };
}

bool IServiceDataFetcher::Fetch(const TFetcherContext& context) {
    if (GetAllowedUserIds().Defined() && GetAllowedUserIds()->empty()
        || GetAllowedCarIds().Defined() && GetAllowedCarIds()->empty()
        || GetAllowedSessionIds().Defined() && GetAllowedSessionIds()->empty()) {
        DEBUG_LOG << context.GetCurrentState() << " skipping fetch because of empty filters" << Endl;
        return true;
    }
    return DoFetch(context);
}

void IServiceDataFetcher::AddAllowedId(TMaybe<TSet<TString>>& dataSet, const TString& id) const {
    if (dataSet.Defined()) {
        dataSet->insert(id);
    } else {
        dataSet = { id };
    }
}

void IServiceDataFetcher::SetEmptyFilters() const {
    SetAllowedUserIds(TSet<TString>());
    SetAllowedCarIds(TSet<TString>());
    SetAllowedSessionIds(TSet<TString>());
}

void IServiceDataFetcher::AddFetchedIdFilters(const IFetchedIterator::TPtr iter, const EAlertEntityType entityType) const {
    TString id(iter->GetObjectId().Data(), iter->GetObjectId().Size());
    switch (entityType) {
    case EAlertEntityType::Car:
        AddAllowedId(MutableAllowedCarIds(), id);
        break;
    case EAlertEntityType::User:
        AddAllowedId(MutableAllowedUserIds(), id);
        break;
    case EAlertEntityType::Session:
        {
            auto userId = iter->GetObjectId(EAlertEntityType::User);
            AddAllowedId(MutableAllowedUserIds(), TString(userId.Data(), userId.Size()));
            auto carId = iter->GetObjectId(EAlertEntityType::Car);
            AddAllowedId(MutableAllowedCarIds(), TString(carId.Data(), carId.Size()));
        }
        AddAllowedId(MutableAllowedSessionIds(), id);
        break;
    default:
        break;
    }
}

IServiceDataFetcher::IServiceDataFetcher(const IServiceDataFetcherConfig& config)
    : BaseConfig(config)
{
    EntityType = config.GetEntityType();
}

TMaybe<TVector<IFetchedIterator::TPtr>> IServiceDataFetcher::BuildIterators(TFetcherContext& context) const {
    TVector<IFetchedIterator::TPtr> result;
    for (auto&& iteratorConfig : BaseConfig.GetIteratorConfigs()) {
        auto iterator = IFetchedIterator::TFactory::Construct(iteratorConfig->GetIteratorType(), BaseConfig.GetFetcherType(), GetEntityType(), context, iteratorConfig);
        if (!iterator || !iterator->Init(*this)) {
            context.AddError("initialization", "incorrect iterator " + ::ToString(iteratorConfig->GetIteratorType()) + " for " + ::ToString(BaseConfig.GetFetcherType()));
            return {};
        }
        if (iterator->GetSubstitutionName()) {
            context.RegisterIterator(iterator->GetField(), iterator->GetSubstitutionName());
        }
        result.push_back(std::move(iterator));
    }
    return result;
}

void TFetcherContext::SaveIteratorData(const TString& objectId, const TString& iteratorName, const TFetchedValue& data) {
    auto nameIt = NamedIterators.find(iteratorName);
    if (nameIt != NamedIterators.end()) {
        TString keyName = ::ToString(nameIt->second);
        IDataIntervalChecker::TPtr formatter =  IDataIntervalChecker::Construct(keyName);
        Y_ENSURE_BT(formatter, "cannot construct DataIntervalChecker for key " << keyName);
        SaveData(objectId, iteratorName, formatter->Format(data, *GetServer(), ELocalization::Rus));
    }
}
