#pragma once

#include "interval_checker.h"

#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>
#include <drive/backend/rt_background/manager/state.h>

#include <drive/library/cpp/common/object_sharding.h>
#include <drive/library/cpp/scheme/scheme.h>

#include <rtline/util/types/accessor.h>
#include <rtline/util/types/messages_collector.h>

#include <util/generic/vector.h>


namespace NAlerts {
    class TFetcherContext;

    class IIteratorConfig {
    public:
        R_FIELD(EFetchedItems, IteratorType, EFetchedItems::Main);
        R_READONLY(TAtomicSharedPtr<IDataIntervalChecker>, Checker);

    public:
        using TFactory = NObjectFactory::TParametrizedObjectFactory<IIteratorConfig, EFetchedItems, EFetchedItems, TAtomicSharedPtr<IDataIntervalChecker>>;
        using TPtr = TAtomicSharedPtr<IIteratorConfig>;

    public:
        IIteratorConfig(EFetchedItems type, TAtomicSharedPtr<IDataIntervalChecker> checker)
            : IteratorType(type)
            , Checker(checker)
        {
        }

        virtual ~IIteratorConfig() = default;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) = 0;
        virtual NJson::TJsonValue SerializeToJson() const = 0;
        virtual NDrive::TScheme GetScheme(const IServerBase& /*server*/) const = 0;
    };

    class IServiceDataFetcher;

    class IFetchedIterator {
    public:
        R_READONLY(EDataFetcherType, Type, EDataFetcherType::ETestFetcher);
        R_READONLY(EAlertEntityType, EntityType, EAlertEntityType::Undefined);

    public:
        using TFactory = NObjectFactory::TParametrizedObjectFactory<IFetchedIterator, EFetchedItems, const EDataFetcherType, const EAlertEntityType, const TFetcherContext&, const IIteratorConfig::TPtr&>;
        using TPtr = TAtomicSharedPtr<IFetchedIterator>;

    public:
        IFetchedIterator(EDataFetcherType type, EAlertEntityType entityType)
            : Type(type)
            , EntityType(entityType)
        {}

        virtual TMaybe<TFetchedValue> Get() = 0;
        virtual TStringBuf GetObjectId() const = 0;
        virtual TStringBuf GetObjectId(const EAlertEntityType entityType) const = 0;
        virtual bool IsFinished() const = 0;
        virtual bool Next() = 0;
        virtual bool Init(const IServiceDataFetcher& /*fetcher*/) = 0;
        virtual EFetchedItems GetField() const = 0;
        virtual IDataIntervalChecker::TPtr GetChecker() const = 0;
        virtual TString GetSubstitutionName() const = 0;
        virtual ~IFetchedIterator() {}

        static TFetchedValue InvalidData() {
            return NAlerts::InvalidData;
        }

        const TString GetFetcherName() const {
            return ::ToString(GetType()) + "." + ::ToString(GetField());
        }
    };

    using TFetchedVec = TVector<TFetchedValue>;
    using TFetchedMap = TMap<TString, TFetchedValue>;
    class IServiceDataFetcherConfig;

    class IServiceDataFetcher {
    public:
        R_FIELD(TMaybe<TSet<TString>>, AllowedUserIds, {}, mutable);
        R_FIELD(TMaybe<TSet<TString>>, AllowedCarIds, {}, mutable);
        R_FIELD(TMaybe<TSet<TString>>, AllowedSessionIds, {}, mutable);
        R_READONLY(EAlertEntityType, EntityType, EAlertEntityType::Undefined);

    public:
        using TPtr = TAtomicSharedPtr<IServiceDataFetcher>;

        IServiceDataFetcher(const IServiceDataFetcherConfig& config);
        bool Fetch(const TFetcherContext& context);
        virtual TMaybe<TVector<IFetchedIterator::TPtr>> BuildIterators(TFetcherContext& context) const;
        virtual void AddFetchedIdFilters(const IFetchedIterator::TPtr iter, const EAlertEntityType entityType) const;
        virtual void SetEmptyFilters() const;
        virtual ~IServiceDataFetcher() {}

    protected:
        virtual bool DoFetch(const TFetcherContext& context) = 0;

    private:
        void AddAllowedId(TMaybe<TSet<TString>>& dataSet, const TString& id) const;

    private:
        const IServiceDataFetcherConfig& BaseConfig;
    };

    class TChainContext {
    public:
        R_READONLY(TSet<TString>, AcceptableStates);
        R_READONLY(TString, StateTagName);
        R_READONLY(TDuration, StateInterval);

    public:
        bool CheckState(const TVector<TString>& statesHistory, const TVector<TInstant>& timeline, const TInstant checkInstant) const;
        bool DeserializeFromJson(const NJson::TJsonValue& json);
        void SerializeToJson(NJson::TJsonValue&) const;
        static void GetScheme(NDrive::TScheme& scheme, const TVector<TString>& alertTags);
    };

    class TFetcherContext {
    public:
        using TFetchers = TMap<NAlerts::EDataFetcherType, IServiceDataFetcher::TPtr>;
        using TRobotStatePtr = TAtomicSharedPtr<IRTBackgroundProcessState>;

    public:
        R_READONLY(TFetchers, DataFetchers);
        R_READONLY(EAlertEntityType, EntityType);
        R_FIELD(TInstant, FetchInstant);
        R_FIELD(TString, RobotUserId);
        R_FIELD(TString, CurrentState);
        R_READONLY(TSet<TString>, ObjectsFilter);
        R_FIELD(TObjectSharding, ShardingPolicy);
        R_OPTIONAL(TChainContext, ChainContext);
        R_READONLY(TMessagesCollector, Errors, {}, mutable);
        R_FIELD(TRobotStatePtr, RobotState);

        using TFetchedData = TMap<TString, TMap<TString, TString>>;
        R_READONLY(TFetchedData, FetchedParameters);

        using TNamedIterators = TMap<TString, NAlerts::EFetchedItems>;
        R_FIELD(TNamedIterators, NamedIterators);

    private:
        const NDrive::IServer* Server = nullptr;
        bool LogEnabled = false;

    public:
        TFetcherContext(const NDrive::IServer* server, EAlertEntityType entityType, bool logEnabled = false)
            : EntityType(entityType)
            , Server(server)
            , LogEnabled(logEnabled)
        {}

        const NDrive::IServer* GetServer() const {
            return Server;
        }

        void AddError(const TString& location, const TString& error) const;

        void AddFetcher(const EDataFetcherType type, IServiceDataFetcher::TPtr fetcher) {
            DataFetchers.insert({ type, fetcher });
        }

        void RegisterIterator(const EFetchedItems iteratorType, const TString& iteratorName) {
            NamedIterators.emplace(iteratorName, iteratorType);
        }

        void SaveData(const TString& objectId, const TString& key, const TString& data) {
            FetchedParameters[objectId].insert({ key,  data });
        }

        void SaveIteratorData(const TString& objectId, const TString& iteratorName, const TFetchedValue& data);

        void SkipObject(const TString& objectId) {
            ObjectsFilter.emplace(objectId);
        }

        bool IsFiltered(const TString& objectId) const {
            return ObjectsFilter.contains(objectId);
        }

        void AddToLog(const TStringBuf& logString) {
            if (LogEnabled) {
                INFO_LOG << "CommonAlertsLog: " << CurrentState << ": " << logString << Endl;
            }
        }

        static TMaybe<NEntityTagsManager::EEntityType> GetTagEntityType(const EAlertEntityType alertType) {
            switch (alertType) {
            case EAlertEntityType::User:
                return NEntityTagsManager::EEntityType::User;
            case EAlertEntityType::Car:
                return NEntityTagsManager::EEntityType::Car;
            case EAlertEntityType::Session:
                return NEntityTagsManager::EEntityType::Trace;
            default:
                return {};
            }
        }
    };

    class IServiceDataFetcherConfig {
    public:
        R_FIELD(i32, Index, 0);
        R_READONLY(TVector<IIteratorConfig::TPtr>, IteratorConfigs);

    public:
        using TFactory = NObjectFactory::TObjectFactory<IServiceDataFetcherConfig, EDataFetcherType>;
        using TPtr = TAtomicSharedPtr<IServiceDataFetcherConfig>;
        using TItems = TMap<NAlerts::EDataFetcherType, TPtr>;

    public:
        NDrive::TScheme GetScheme(const IServerBase& server) const;
        bool DeserializeFromJson(const NJson::TJsonValue& json);
        NJson::TJsonValue SerializeToJson() const;

        virtual bool DoDeserializeFromJson(const NJson::TJsonValue& json) = 0;
        virtual NJson::TJsonValue DoSerializeToJson() const = 0;
        virtual NDrive::TScheme DoGetScheme(const IServerBase& /*server*/) const = 0;

        virtual EDataFetcherType GetFetcherType() const = 0;
        virtual TString GetSchemeDescription() const = 0;
        virtual EAlertEntityType GetEntityType() const = 0;
        virtual TSet<EAlertEntityType> GetAcceptableEntityTypes() const;
        virtual IServiceDataFetcher::TPtr BuildFetcher() const = 0;
        virtual ~IServiceDataFetcherConfig() {}

        void AddIteratorConfig(const IIteratorConfig::TPtr& iteratorConfig) {
            IteratorConfigs.push_back(iteratorConfig);
        }

        static TPtr BuildFromJson(const NJson::TJsonValue& json) {
            TString typeStr = json["type"].GetString();
            EDataFetcherType type;
            if (!TryFromString(typeStr, type)) {
                return nullptr;
            }

            THolder<IServiceDataFetcherConfig> config = THolder(TFactory::Construct(type));
            if (!config || !config->DeserializeFromJson(json)) {
                return nullptr;
            }
            return config.Release();
        }
    };
}
