#include "chat.h"

#include <drive/backend/chat_robots/ifaces.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/user_devices/manager.h>
#include <drive/backend/user_document_photos/manager.h>

TChatUsersFetcherConfig::TFactory::TRegistrator<TChatUsersFetcherConfig> TChatUsersFetcherConfig::Registrator(NAlerts::EDataFetcherType::EChatState);

bool TChatUsersFetcherConfig::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_STRING_OPT(json, "chat_id", ChatId);
    JREAD_STRING_OPT(json, "chat_topic", ChatTopic);
    JREAD_BOOL_OPT(json, "exclude_active_twins", ExcludeActiveTwins);
    JREAD_STRING_OPT(json, "base_status", BaseStatus);
    if (!TJsonProcessor::ReadContainer(json, "node_names", NodeNames)) {
        return false;
    }
    if (json.Has("photo_statuses")) {
        if (!json["photo_statuses"].IsArray()) {
            ERROR_LOG << "photo_statuses are not an array" << Endl;
            return false;
        }
        for (auto&& item : json["photo_statuses"].GetArray()) {
            if (!item.IsMap()) {
                ERROR_LOG << "photo_statuses item is not map" << Endl;
                return false;
            }
            NUserDocument::EType photoType;
            NUserDocument::EVerificationStatus vStatus;
            JREAD_FROM_STRING(item, "type", photoType);
            JREAD_FROM_STRING(item, "status", vStatus);
            PhotoStatuses[photoType] = vStatus;
        }
    }
    if (json.Has("cv_filter_blocks")) {
        if (!json["cv_filter_blocks"].IsArray()) {
            ERROR_LOG << "Alert parsing failed: photoConfArray is not an array" << Endl;
            return false;
        }
        auto photoConfArray = json["cv_filter_blocks"].GetArray();
        for (auto&& blockJson : photoConfArray) {
            TChatUsersFetcherConfig::TConfidencesBlock block;
            if (!block.DeserializeFromJson(blockJson)) {
                return false;
            }
            CVFilterBlocks.emplace_back(std::move(block));
        }
    }
    return true;
}

bool TChatUsersFetcherConfig::TConfidencesBlock::DeserializeFromJson(const NJson::TJsonValue& json) {
    if (!json.Has("fields") || !json["fields"].IsArray()) {
        return false;
    }
    for (auto&& fieldStr : json["fields"].GetArray()) {
        TRecognitionConfidenceData::EDocumentField field;
        if (!TryFromString(fieldStr.GetString(), field)) {
            return false;
        }
        Fields.push_back(field);
    }
    if (Fields.empty()) {
        return false;
    }
    JREAD_FROM_STRING(json, "photo_type", PhotoType);
    JREAD_DOUBLE_OPT(json, "min_avg_confidence", MinAvgConfidence);
    JREAD_DOUBLE_OPT(json, "min_each_confidence", MinEachConfidence);
    return true;
}

NJson::TJsonValue TChatUsersFetcherConfig::TConfidencesBlock::SerializeToJson() const {
    NJson::TJsonValue result;
    {
        NJson::TJsonValue fieldsJson = NJson::JSON_ARRAY;
        for (auto&& field : Fields) {
            fieldsJson.AppendValue(ToString(field));
        }
        result["fields"] = std::move(fieldsJson);
    }
    result["photo_type"] = ToString(PhotoType);
    result["min_avg_confidence"] = MinAvgConfidence;
    result["min_each_confidence"] = MinEachConfidence;
    return result;
}

NJson::TJsonValue TChatUsersFetcherConfig::DoSerializeToJson() const {
    NJson::TJsonValue result;
    JWRITE(result, "chat_id", ChatId);
    JWRITE(result, "chat_topic", ChatTopic);
    JWRITE(result, "exclude_active_twins", ExcludeActiveTwins);
    JWRITE(result, "base_status", BaseStatus);
    TJsonProcessor::WriteContainerArray(result, "node_names", NodeNames);
    if (PhotoStatuses.size()) {
        NJson::TJsonValue photoStatusesArray = NJson::JSON_ARRAY;
        for (auto&& it : PhotoStatuses) {
            NJson::TJsonValue photoStatus;
            photoStatus["type"] = ToString(it.first);
            photoStatus["status"] = ToString(it.second);
            photoStatusesArray.AppendValue(std::move(photoStatus));
        }
        result["photo_statuses"] = std::move(photoStatusesArray);
    }
    {
        NJson::TJsonValue cvFilterBlocks = NJson::JSON_ARRAY;
        for (auto&& block : CVFilterBlocks) {
            cvFilterBlocks.AppendValue(block.SerializeToJson());
        }
        result["cv_filter_blocks"] = std::move(cvFilterBlocks);
    }
    return result;
}

NDrive::TScheme TChatUsersFetcherConfig::DoGetScheme(const IServerBase& /*server*/) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("chat_id", "id чата");
    scheme.Add<TFSString>("chat_topic", "Топик");
    scheme.Add<TFSBoolean>("exclude_active_twins", "Исключать активных двойников");
    scheme.Add<TFSArray>("node_names", "Имена состояний в чате").SetElement<TFSString>();
    scheme.Add<TFSString>("base_status", "Статус пользователей для выборки");

    {
        NDrive::TScheme itemScheme;
        itemScheme.Add<TFSVariants>("type", "Тип фото").InitVariants<NUserDocument::EType>();
        itemScheme.Add<TFSVariants>("status", "Статус фото").InitVariants<NUserDocument::EVerificationStatus>();
        scheme.Add<TFSArray>("photo_statuses", "Статусы фото").SetElement(itemScheme);
    }

    {
        NDrive::TScheme itemScheme;
        itemScheme.Add<TFSVariants>("photo_type", "Тип фото").InitVariants<NUserDocument::EType>();
        itemScheme.Add<TFSVariants>("fields", "Поля").InitVariants<TRecognitionConfidenceData::EDocumentField>().SetMultiSelect(true);
        itemScheme.Add<TFSNumeric>("min_avg_confidence", "Минимально допустимый средний confidence по полям");
        itemScheme.Add<TFSNumeric>("min_each_confidence", "Минимально допустимый confidence по каждому полю");
        scheme.Add<TFSArray>("cv_filter_blocks", "Фильтры по распознаванию").SetElement(itemScheme);
    }

    return scheme;
}

NAlerts::IServiceDataFetcher::TPtr TChatUsersFetcherConfig::BuildFetcher() const {
    return new TChatUsersFetcher(*this);
}

bool TChatUsersFetcher::DoFetch(const NAlerts::TFetcherContext& context) {
    if (!Users.empty()) {
        return false;
    }

    auto chatRobot = context.GetServer()->GetChatRobot(Config.GetChatId());
    if (!chatRobot) {
        return false;
    }
    if (!chatRobot->RefreshStates(context.GetFetchInstant())) {
        return false;
    }

    TSet<TString> userIds;
    auto usersInNodes = chatRobot->GetUsersInNodes(Config.GetChatTopic(), Config.GetNodeNames());
    for (auto&& userIt : usersInNodes) {
        userIds.emplace(std::move(userIt.first));
    }

    auto usersData = context.GetServer()->GetDriveAPI()->GetUsersData();
    if (!usersData) {
        return false;
    }

    {
        auto usersFetchResult = usersData->FetchInfo(userIds);
        for (auto&& userIt : usersFetchResult) {
            if (userIt.second.GetStatus() != Config.GetBaseStatus()) {
                userIds.erase(userIt.first);
            }
        }
    }

    if (Config.GetExcludeActiveTwins()) {
        TUsersDB::TFetchResult extendedUsersFetchResult;

        // exclude active twins by device id
        {
            auto session = context.GetServer()->GetDriveAPI()->BuildSession(true, false);

            TSet<TShortUserDevice> chatUsersDevices;
            auto manager = context.GetServer()->GetUserDevicesManager();
            if (!manager || !manager->GetUserDevices(userIds, chatUsersDevices, session)) {
                return false;
            }

            TSet<TString> deviceIds;
            for (auto&& device : chatUsersDevices) {
                deviceIds.insert(device.GetDeviceId());
            }

            TSet<TString> extendedUserIds;
            if (!manager->GetUsersByDeviceIds(deviceIds, extendedUserIds, session)) {
                return false;
            }

            TSet<TString> activeTwinUserIds;
            extendedUsersFetchResult = usersData->FetchInfo(extendedUserIds);
            for (auto&& userIt : extendedUsersFetchResult) {
                if (userIt.second.GetStatus() != Config.GetBaseStatus() && !userIds.contains(userIt.first)) {
                    activeTwinUserIds.insert(userIt.first);
                }
            }

            TSet<TShortUserDevice> activeTwinDevices;
            if (!context.GetServer()->GetUserDevicesManager()->GetUserDevices(activeTwinUserIds, activeTwinDevices, session)) {
                return false;
            }
            TSet<TString> bannedDeviceIds;
            for (auto&& device : activeTwinDevices) {
                bannedDeviceIds.insert(device.GetDeviceId());
            }

            for (auto&& device : chatUsersDevices) {
                if (bannedDeviceIds.contains(device.GetDeviceId())) {
                    userIds.erase(device.GetUserId());
                }
            }
        }

        // exclude active twins by phone number
        {
            TSet<TString> phoneNumbers;
            TSet<TString> bannedUserIds;
            for (auto&& userId : userIds) {
                auto userPtr = extendedUsersFetchResult.GetResultPtr(userId);
                if (!userPtr) {
                    continue;
                }
                if (!userPtr->IsPhoneVerified()) {
                    bannedUserIds.insert(userId);
                }
                phoneNumbers.insert(userPtr->GetPhone());
            }

            TSet<TString> bannedPhoneNumbers;
            auto phoneTwinsFetchResult = usersData->FetchUsersByPhone(phoneNumbers);
            for (auto&& userIt : phoneTwinsFetchResult) {
                if (userIt.second.GetStatus() != Config.GetBaseStatus() && !userIds.contains(userIt.first)) {
                    bannedPhoneNumbers.insert(userIt.second.GetPhone());
                }
            }
            for (auto&& userIt : phoneTwinsFetchResult) {
                if (bannedPhoneNumbers.contains(userIt.second.GetPhone())) {
                    userIds.erase(userIt.first);
                }
            }
            for (auto&& userId : bannedUserIds) {
                userIds.erase(userId);
            }
        }
    }

    if (Config.GetPhotoStatuses().size()) {
        if (!context.GetServer()->GetDriveAPI()->HasDocumentPhotosManager()) {
            return false;
        }
        TSet<TString> filteredUsers;

        TSet<NUserDocument::EType> targetPhotoTypes;
        for (auto&& type : NContainer::Keys(Config.GetPhotoStatuses())) {
            targetPhotoTypes.insert(type);
        }
        for (auto&& block : Config.GetCVFilterBlocks()) {
            targetPhotoTypes.insert(block.GetPhotoType());
        }

        auto usersWithPhotos = context.GetServer()->GetDriveAPI()->GetDocumentPhotosManager().GetUserPhotosDB().GetTypeToRecentPhotoMapping(userIds, MakeVector(targetPhotoTypes));
        for (auto&& [userId, photosMap] : usersWithPhotos) {
            if (photosMap.size() < Config.GetPhotoStatuses().size()) {
                continue;
            }
            bool mismatch = false;
            for (auto&& [photoType, photoSoughtStatus] : Config.GetPhotoStatuses()) {
                auto photoIt = photosMap.find(photoType);
                if (photoIt == photosMap.end() || photoIt->second.GetVerificationStatus() != photoSoughtStatus) {
                    mismatch = true;
                    break;
                }
            }
            if (mismatch) {
                continue;
            }
            for (auto&& block : Config.GetCVFilterBlocks()) {
                Y_ENSURE_BT(!block.GetFields().empty());
                auto photoIt = photosMap.find(block.GetPhotoType());
                if (photoIt == photosMap.end() || !photoIt->second.HasRecognizerMeta()) {
                    mismatch = true;
                    break;
                }
                double sumConf = 0;
                for (auto&& field : block.GetFields()) {
                    auto conf = photoIt->second.GetRecognizerMetaUnsafe().GetField(field);
                    if (conf < block.GetMinEachConfidence()) {
                        mismatch = true;
                        break;
                    }
                    sumConf += conf;
                }
                if (mismatch || sumConf / block.GetFields().size() < block.GetMinAvgConfidence()) {
                    mismatch = true;
                    break;
                }
            }
            if (!mismatch) {
                filteredUsers.insert(userId);
            }
        }
        userIds = std::move(filteredUsers);
    }

    auto usersFetchResult = usersData->FetchInfo(userIds);
    for (auto userIt : usersFetchResult) {
        if (!GetAllowedUserIds().Defined() || GetAllowedUserIds()->contains(userIt.second.GetUserId())) {
            Users.emplace_back(std::move(userIt.second));
        }
    }

    return true;
}
