#include "object_tags.h"

#include <drive/backend/data/markers.h>
#include <drive/backend/database/drive_api.h>

TTagsFetcherConfig::TFactory::TRegistrator<TTagsFetcherConfig> TTagsFetcherConfig::Registrator(TTagsFetcherConfig::GetType());

bool TTagsFetcherConfig::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    if (!SearchRequest.DeserializeFromJson(json["filter"])) {
        return false;
    }
    JREAD_FROM_STRING_OPT(json, "entity", EntityType);
    if (!ChainContext.DeserializeFromJson(json)) {
        return false;
    }
    return true;
}

NJson::TJsonValue TTagsFetcherConfig::DoSerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "entity", NJson::Stringify(EntityType));
    JWRITE(result, "filter", SearchRequest.SerializeToJson());
    ChainContext.SerializeToJson(result);
    return result;
}

NDrive::TScheme TTagsFetcherConfig::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme;
    {
        auto gTab = scheme.StartTabGuard("TagsFilter");
        scheme.Add<TFSStructure>("filter", "Фильтр объектов").SetStructure(TTagsSearchRequest::GetScheme(server.GetAsSafe<NDrive::IServer>()));
        TVector<TString> entities = { ToString(NAlerts::EAlertEntityType::User), ToString(NAlerts::EAlertEntityType::Car), ToString(NAlerts::EAlertEntityType::Session) };
        scheme.Add<TFSVariants>("entity", "Сущность").SetVariants(entities).SetMultiSelect(false);
    }
    {
        auto gTab = scheme.StartTabGuard("ChainLogic");
        ITagsMeta::TTagDescriptionsByName tags= server.GetAsSafe<NDrive::IServer>().GetDriveAPI()->GetTagsManager().GetTagsMeta().GetRegisteredTags(NEntityTagsManager::EEntityType::User, { TUserAlertTag::TypeName, TCarAlertTag::TypeName });
        NAlerts::TChainContext::GetScheme(scheme, MakeVector(NContainer::Keys(tags)));
    }
    return scheme;
}

TString TTagsFetcherConfig::GetSchemeDescription() const {
    return "Фильтрация объектов по тегам";
}

NAlerts::EDataFetcherType TTagsFetcherConfig::GetFetcherType() const {
    return GetType();
}

NAlerts::IServiceDataFetcher::TPtr TTagsFetcherConfig::BuildFetcher() const {
    return new TTagsDataFetcher(*this);
}

bool TTagsDataFetcher::DoFetch(const NAlerts::TFetcherContext& context) {
    if (SearchResult.Defined()) {
        return true;
    }
    auto tagEntity = NAlerts::TFetcherContext::GetTagEntityType(GetEntityType());
    if (!tagEntity.Defined()) {
        return false;
    }
    auto search = context.GetServer()->GetDriveAPI()->GetTagsSearch(tagEntity.GetRef());
    if (!search) {
        return false;
    }
    TTagsSearchRequest req = Config.GetSearchRequest();
    SearchResult = search->Search(req);
    TSet<TString> objectsIds;
    if (!Config.GetChainContext().GetStateTagName()) {
        objectsIds = MakeSet(NContainer::Keys(SearchResult->GetAssociatedTags()));
    } else {
        for (auto&& object : SearchResult->GetAssociatedTags()) {
            bool accept = false;
            for (auto& tag : object.second) {
                if (tag->GetName() == Config.GetChainContext().GetStateTagName()) {
                    const TAlertTag* tagData = dynamic_cast<const TAlertTag*>(tag.GetData().Get());
                    if (!tagData) {
                        return false;
                    }
                    if (Config.GetChainContext().CheckState(tagData->GetStatesHistory(), tagData->GetStatesTs(), context.GetFetchInstant())) {
                        accept = true;
                    }
                }
            }
            if (accept) {
                objectsIds.emplace(object.first);
            }
        }
    }
    if (GetEntityType() == NAlerts::EAlertEntityType::User && GetAllowedUserIds().Defined()) {
        std::set_intersection(GetAllowedUserIds()->begin(), GetAllowedUserIds()->end(),
            objectsIds.begin(), objectsIds.end(),
            std::inserter(ObjectsIds, ObjectsIds.begin()));
    } else if (GetEntityType() == NAlerts::EAlertEntityType::Car && GetAllowedCarIds().Defined()) {
        std::set_intersection(GetAllowedCarIds()->begin(), GetAllowedCarIds()->end(),
            objectsIds.begin(), objectsIds.end(),
            std::inserter(ObjectsIds, ObjectsIds.begin()));
    } else if (GetEntityType() == NAlerts::EAlertEntityType::Session && GetAllowedSessionIds()) {
        std::set_intersection(GetAllowedSessionIds()->begin(), GetAllowedSessionIds()->end(),
            objectsIds.begin(), objectsIds.end(),
            std::inserter(ObjectsIds, ObjectsIds.begin()));
    } else {
        ObjectsIds = std::move(objectsIds);
    }
    return true;
}
