#include "sensor_history.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/database/transaction/assert.h>

#include <drive/telematics/api/sensor/history.h>

bool TSensorHistoryIdFetcherConfig::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    return
        NJson::ParseField(json["query"], Query) &&
        NJson::ParseField(json["timeout"], Timeout);
}

NJson::TJsonValue TSensorHistoryIdFetcherConfig::DoSerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "query", Query);
    NJson::InsertField(result, "timeout", NJson::Hr(Timeout));
    return result;
}

NDrive::TScheme TSensorHistoryIdFetcherConfig::DoGetScheme(const IServerBase& server) const {
    Y_UNUSED(server);
    NDrive::TScheme scheme;
    scheme.Add<TFSString>("query", "SQL query");
    scheme.Add<TFSDuration>("timeout", "Query timeout").SetDefault(Timeout);
    return scheme;
}

NAlerts::IServiceDataFetcher::TPtr TSensorHistoryIdFetcherConfig::BuildFetcher() const {
    return MakeAtomicShared<TSensorHistoryIdFetcher>(*this);
}

bool TSensorHistoryIdFetcher::DoFetch(const NAlerts::TFetcherContext& context) {
    if (!ObjectIds.empty()) {
        return true;
    }

    auto server = context.GetServer();
    auto client = Yensured(server)->GetSensorHistoryClient();
    Y_ENSURE(client);
    auto impl = dynamic_cast<const NDrive::TSensorHistoryClient*>(client);
    Y_ENSURE(impl);
    auto start = Now();
    auto deadline = start + Config.GetTimeout();
    auto result = impl->GetClickHouseClient().SelectIds(Config.GetQuery(), deadline);
    R_ENSURE(result.Wait(deadline), HTTP_INTERNAL_SERVER_ERROR, "wait timeout after " << deadline);
    for (auto&& id : result.GetValue()) {
        const auto& allowedCarIds = GetAllowedCarIds();
        if (allowedCarIds && !allowedCarIds->contains(id)) {
            continue;
        }
        auto object = server->GetDriveDatabase().GetCarManager().GetObject(id);
        if (!object) {
            continue;
        }
        ObjectIds.insert(id);
    }

    return true;
}

TSensorHistoryIdFetcherConfig::TFactory::TRegistrator<TSensorHistoryIdFetcherConfig> TSensorHistoryIdFetcherConfig::Registrator(TSensorHistoryIdFetcherConfig::GetType());
