#include "trace_tags_history.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/tags/tags_manager.h>

#include <drive/backend/history_iterator/history_iterator.h>
#include <drive/backend/rt_background/common_alerts/process.h>

TTraceTagsHistoryConfig::TFactory::TRegistrator<TTraceTagsHistoryConfig> TTraceTagsHistoryConfig::Registrator(NAlerts::EDataFetcherType::ETraceTagsHistory);

bool TTraceTagsHistoryConfig::TQueryOptions::DeserializeFromJson(const NJson::TJsonValue& json) {
    return
        NJson::ParseField(json, "actions", MutableActions()) && HasActions() &&
        NJson::ParseField(json, "performer_ids", MutablePerformers()) &&
        NJson::ParseField(json, "tag_names", MutableTags()) &&
        NJson::ParseField(json, "tag_ids", MutableTagIds()) &&
        NJson::ParseField(json, "user_ids", MutableUserIds()) &&
        NJson::ParseField<bool&>(json, "descending", Descending);
}
NJson::TJsonValue TTraceTagsHistoryConfig::TQueryOptions::SerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertNonNull(result, "actions", Actions);
    NJson::InsertNonNull(result, "performer_ids", Performers);
    NJson::InsertNonNull(result, "tag_names", Tags);
    NJson::InsertNonNull(result, "tag_ids", TagIds);
    NJson::InsertNonNull(result, "user_ids", UserIds);
    NJson::InsertField(result, "descending", Descending);
    return result;
}

NDrive::TScheme TTraceTagsHistoryConfig::TQueryOptions::GetScheme(const IServerBase& /*server*/) {
    NDrive::TScheme scheme;
    scheme.Add<TFSVariants>("actions", "Действия").SetVariants(
        {
            EObjectHistoryAction::Add,
            EObjectHistoryAction::Remove,
            EObjectHistoryAction::UpdateData,
            EObjectHistoryAction::TagEvolve
        }
    ).SetMultiSelect(true);
    scheme.Add<TFSArray>("performer_ids").SetElement<TFSString>();
    scheme.Add<TFSArray>("tag_names").SetElement<TFSString>();
    scheme.Add<TFSArray>("tag_ids").SetElement<TFSString>();
    scheme.Add<TFSArray>("user_ids").SetElement<TFSString>();
    scheme.Add<TFSBoolean>("descending").SetDefault(false).SetReadOnly(true);

    return scheme;
}

TTagEventsManager::TQueryOptions TTraceTagsHistoryConfig::TQueryOptions::GetQueryRequest() const {
    TTagEventsManager::TQueryOptions tagsHistoryQuery;
    if (Actions.Defined()) {
        tagsHistoryQuery.SetActions(GetActionsRef());
    }
    if (Performers.Defined()) {
        tagsHistoryQuery.SetPerformers(GetPerformersRef());
    }
    if (Tags.Defined()) {
        tagsHistoryQuery.SetTags(GetTagsRef());
    }
    if (TagIds.Defined()) {
        tagsHistoryQuery.SetTagIds(GetTagIdsRef());
    }
    if (UserIds.Defined()) {
        tagsHistoryQuery.SetUserIds(GetUserIdsRef());
    }
    tagsHistoryQuery.SetDescending(GetDescending());
    return tagsHistoryQuery;
}

template <>
NJson::TJsonValue NJson::ToJson(const TTraceTagsHistoryConfig::TQueryOptions& object) {
    return object.SerializeToJson();
}

template <>
bool NJson::TryFromJson(const NJson::TJsonValue& value, TTraceTagsHistoryConfig::TQueryOptions& result) {
    return result.DeserializeFromJson(value);
}

bool TTraceTagsHistoryConfig::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    return
        NJson::ParseField(json, "query", QueryOptions) &&
        NJson::ParseField(json, "substruct_query", SubtractQueryOptions) &&
        (!json.Has("cars_filter") || CarsFilter.ConstructInPlace().DeserializeFromJson(json["cars_filter"])) &&
        NJson::ParseField(json, "trace_filter", TraceFilter, true);
}

NJson::TJsonValue TTraceTagsHistoryConfig::DoSerializeToJson() const {
    NJson::TJsonValue result;
    NJson::InsertField(result, "query", QueryOptions);
    NJson::InsertNonNull(result, "substruct_query", SubtractQueryOptions);
    if (HasCarsFilter()) {
        result["cars_filter"] = CarsFilter->SerializeToJson();
    }
    result["trace_filter"] = TraceFilter;
    return result;
}

NDrive::TScheme TTraceTagsHistoryConfig::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSStructure>("query").SetStructure(TTraceTagsHistoryConfig::TQueryOptions::GetScheme(server));
    scheme.Add<TFSStructure>("substruct_query").SetStructure(TTraceTagsHistoryConfig::TQueryOptions::GetScheme(server));
    scheme.Add<TFSStructure>("cars_filter", "Фильтр по машинам").SetStructure(TCarsFilter::GetScheme(server.GetAsSafe<NDrive::IServer>()));
    scheme.Add<TFSString>("trace_filter", "Фильтр тегов по сессиям").SetRequired(true);
    return scheme;
}

NAlerts::IServiceDataFetcher::TPtr TTraceTagsHistoryConfig::BuildFetcher() const {
    return MakeAtomicShared<TTraceTagsHistoryFetcher>(*this);
}

namespace {
    void ActializeEventId(const IBaseSequentialTableImpl::TEventId& maxEventId, TAtomicSharedPtr<TRTCommonAlertsState> stateImpl) {
        if (stateImpl->GetLastEventId() == NDrive::IncorrectEventId || maxEventId > stateImpl->GetLastEventId()) {
            stateImpl->MutableLastEventId() = maxEventId;
        }
    }
}

bool TTraceTagsHistoryFetcher::DoFetch(const NAlerts::TFetcherContext& context) {
    auto* server = context.GetServer();
    if (!server) {
        return false;
    }
    if (!ObjectIds.empty()) {
        return false;
    }
    TAtomicSharedPtr<TRTCommonAlertsState> stateImpl = std::dynamic_pointer_cast<TRTCommonAlertsState>(context.GetRobotState());
    if (!stateImpl) {
        context.AddError("TTraceTagsHistoryFetcher::DoFetch", "stateImpl == nullptr");
        return false;
    }
    TRange<TInstant> timeRange;
    auto startEventCursor = stateImpl->GetLastEventId();
    auto maxEventId = server->GetDriveAPI()->GetTagsManager().GetTraceTags().GetHistoryManager().GetLockedMaxEventId();
    if (maxEventId == startEventCursor) {
        return true;
    }
    if (startEventCursor == NDrive::IncorrectEventId) {
        timeRange.From = context.GetFetchInstant();
        startEventCursor = 0;
    } else {
        startEventCursor++;
    }

    auto tx = server->GetDriveAPI()->template BuildTx<NSQL::Writable>();

    auto eventRequest = [&](const TTagEventsManager::TQueryOptions& tagsHistoryQuery, TVector<TObjectEvent<TConstDBTag>>& resultEvents) -> bool {
        auto optionalEvents = server->GetDriveAPI()->GetTagsManager().GetTraceTags().GetEvents(
              { startEventCursor, maxEventId + 1}
            , timeRange
            , tx
            , tagsHistoryQuery
        );
        if (!optionalEvents) {
            context.AddError("TTraceTagsHistoryFetcher::DoFetch", "GetEvents not response");
            return false;
        }
        resultEvents = std::move(*optionalEvents);
        return true;
    };

    TVector<TObjectEvent<TConstDBTag>> tagsHistoryEvents;
    if (!eventRequest(Config.GetQueryOptions().GetQueryRequest(), tagsHistoryEvents)) {
        return false;
    }
    TVector<TString> sessionIds;
    Transform(tagsHistoryEvents.begin(), tagsHistoryEvents.end(), std::inserter(sessionIds, sessionIds.begin()), [](const auto& tag) { return tag.GetObjectId(); });

    ActializeEventId(maxEventId, stateImpl);

    TSet<TString> sessionIdsSubstruct;
    TVector<TObjectEvent<TConstDBTag>> substructTagsHistoryEvents;
    if (Config.HasSubtractQueryOptions()) {
        TTagEventsManager::TQueryOptions tagsHistoryQuery = Config.GetQueryOptions().GetQueryRequest();
        if (!eventRequest(Config.GetSubtractQueryOptionsRef().GetQueryRequest(), substructTagsHistoryEvents)) {
            return false;
        }
        TSet<TString> sessionIds;
        Transform(substructTagsHistoryEvents.begin(), substructTagsHistoryEvents.end(), std::inserter(sessionIdsSubstruct, sessionIdsSubstruct.begin()), [](const auto& tag) { return tag.GetObjectId(); });
        sessionIds = MakeIntersection(sessionIds, sessionIdsSubstruct);
    }

    if (sessionIds.empty()) {
        return true;
    }

    auto ydbTx = server->GetDriveAPI()->BuildYdbTx<NSQL::ReadOnly>("TTraceTagsHistoryFetcher::DoFetch", server);
    THistoryRidesContext sessionsContext(*server, TInstant::Zero(), true);
    if (!sessionsContext.InitializeSessions(sessionIds, tx, ydbTx)) {
        context.AddError("TTraceTagsHistoryFetcher::DoFetch", "cannot InitializeSessions");
        return false;
    }

    auto rideFilter = [&config = Config](const THistoryRideObject& ride) -> bool {
        auto filter = TTagsFilter::BuildFromString(config.GetTraceFilter());
        return ride.GetTraceTags() && filter.IsMatching(*ride.GetTraceTags());
    };

    TSet<TString> filteredCarIds;
    if (Config.HasCarsFilter() && !Config.GetCarsFilterRef().GetAllowedCarIds(filteredCarIds, context.GetServer(), context.GetFetchInstant())) {
        return false;
    }

    bool hasMore = true;
    auto rides = sessionsContext.GetSessions(TInstant::Now(), 100000, &hasMore, false, rideFilter);
    for (const auto& ride : rides) {
        if (filteredCarIds.empty() || filteredCarIds.contains(ride.GetObjectId())) {
            ObjectIds.insert(ride.GetSessionId());
        }
    }
    return tx.Rollback();
}
