#include "user.h"

#include <drive/backend/database/drive_api.h>
#include <drive/backend/roles/manager.h>

TUsersFetcherConfig::TFactory::TRegistrator<TUsersFetcherConfig> TUsersFetcherConfig::Registrator(NAlerts::EDataFetcherType::EUserData);

bool TUsersFetcherConfig::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    JREAD_BOOL_OPT(json, "check_ridings", CheckRidings);
    JREAD_BOOL_OPT(json, "without_ridings", WithoutRidings);
    if (!TJsonProcessor::ReadContainer(json, "user_status", Statuses)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(json, "user_actions", Actions)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(json, "user_roles", Roles)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(json, "special_users", SpecialUsersList)) {
        return false;
    }
    if (!TJsonProcessor::ReadContainer(json, "reg_geo", RegistrationGeo)) {
        return false;
    }
    return true;
}

NJson::TJsonValue TUsersFetcherConfig::DoSerializeToJson() const {
    NJson::TJsonValue result;
    JWRITE(result, "check_ridings", CheckRidings);
    JWRITE(result, "without_ridings", WithoutRidings);
    TJsonProcessor::WriteContainerArray(result, "user_status", Statuses);
    TJsonProcessor::WriteContainerArray(result, "user_actions", Actions);
    TJsonProcessor::WriteContainerArray(result, "user_roles", Roles);
    TJsonProcessor::WriteContainerArray(result, "special_users", SpecialUsersList);
    TJsonProcessor::WriteContainerArray(result, "reg_geo", RegistrationGeo);
    return result;
}

NDrive::TScheme TUsersFetcherConfig::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme;
    scheme.Add<TFSBoolean>("check_ridings", "Проверять колличество поездок").SetDefault(true);
    scheme.Add<TFSBoolean>("without_ridings", "Брать только пользователей без поездок");
    scheme.Add<TFSArray>("user_status", "Фильтр состояний").SetElement<TFSString>();
    scheme.Add<TFSArray>("special_users", "Выбрать только этих пользователей").SetElement<TFSString>();
    scheme.Add<TFSArray>("reg_geo", "Гео пользователя").SetElement<TFSString>();
    auto impl = server.GetAs<NDrive::IServer>();
    if (impl) {
        scheme.Add<TFSVariants>("user_actions", "Фильтр экшенов").SetVariants(impl->GetDriveAPI()->GetRolesManager()->GetActionsDB().GetCachedObjectsMap()).SetMultiSelect(true);
        scheme.Add<TFSVariants>("user_roles", "Фильтр ролей").SetVariants(impl->GetDriveAPI()->GetRolesManager()->GetRolesDB().GetCachedObjectsMap()).SetMultiSelect(true);
    } else {
        scheme.Add<TFSArray>("user_actions", "Фильтр экшенов").SetElement<TFSString>();
        scheme.Add<TFSArray>("user_roles", "Фильтр ролей").SetElement<TFSString>();
    }
    return scheme;
}


NAlerts::IServiceDataFetcher::TPtr TUsersFetcherConfig::BuildFetcher() const {
    return new TUsersFetcher(*this);
}

bool TUsersFetcher::DoFetch(const NAlerts::TFetcherContext& context) {
    if (!Users.empty()) {
        return false;
    }

    auto usersData = context.GetServer()->GetDriveAPI()->GetUsersData();
    if (!usersData) {
        return false;
    }

    bool allUsers = true;
    TSet<TString> userIds;
    if (!Config.GetActions().empty() || !Config.GetRoles().empty()) {
        if (!context.GetServer()->GetDriveAPI()->GetRolesManager()) {
            return false;
        }
        auto roles = context.GetServer()->GetDriveAPI()->GetRolesManager()->GetRolesWithActions(Config.GetActions(), context.GetFetchInstant());
        TSet<TString> roleIds = Config.GetRoles();
        for (const auto& role : roles) {
            roleIds.emplace(role.GetName());
        }

        allUsers = false;
        if (!usersData->GetRoles().GetUsersWithAtLeastOneRoles(roleIds, userIds, true)) {
            ERROR_LOG << "GetUsersWithAtLeastOneRoles failed! returning empty set" << Endl;
            return false;
        }
    }

    if (!Config.GetSpecialUsersList().empty()) {
        if (allUsers) {
            userIds = Config.GetSpecialUsersList();
        } else {
            for (auto it = userIds.begin(); it != userIds.end();) {
                if (!Config.GetSpecialUsersList().contains(*it)) {
                    it = userIds.erase(it);
                } else {
                    ++it;
                }
            }
        }
        allUsers = false;
    }

    TUsersDB::TFetchResult fetched;
    if (allUsers) {
        NSQL::TQueryOptions queryOptions;
        if (!Config.GetStatuses().empty()) {
            queryOptions.SetGenericCondition("status", Config.GetStatuses());
        }
        if (!Config.GetRegistrationGeo().empty()) {
            queryOptions.SetGenericCondition("registration_geo", Config.GetRegistrationGeo());
        }
        if (Config.IsCheckRidings()) {
            queryOptions.SetGenericCondition("is_first_riding", Config.GetWithoutRidings());
        }
        if (GetAllowedUserIds()) {
            queryOptions.SetGenericCondition("id", *GetAllowedUserIds());
        }
        auto session = usersData->BuildSession(true);
        auto entities = usersData->Fetch(session, queryOptions);
        if (!entities) {
            session.Check();
        }
        for (auto&& entity : *entities) {
            auto id = entity.GetUserId();
            fetched.AddResult(id, std::move(entity));
        }
    } else {
        fetched = usersData->FetchInfo(userIds);
    }
    for (auto&& entity : fetched) {
        if (!Config.GetStatuses().empty() && !Config.GetStatuses().contains(entity.second.GetStatus())) {
            continue;
        }
        if (!Config.GetRegistrationGeo().empty() && !Config.GetRegistrationGeo().contains(entity.second.GetRegistrationGeo())) {
            continue;
        }
        if (Config.IsCheckRidings() && !(Config.GetWithoutRidings() && entity.second.IsFirstRiding()) &&
            !(!Config.GetWithoutRidings() && !entity.second.IsFirstRiding())) {
            continue;
        }

        if (!GetAllowedUserIds().Defined() || GetAllowedUserIds()->contains(entity.second.GetUserId())) {
            Users.emplace_back(std::move(entity.second));
        }
    }
    return true;
}
