#include "user_sessions.h"

#include <drive/backend/cars/car.h>
#include <drive/backend/cars/car_model.h>
#include <drive/backend/database/drive_api.h>
#include <drive/backend/offers/action.h>
#include <drive/backend/roles/manager.h>

TSessionsFetcherConfig::TFactory::TRegistrator<TSessionsFetcherConfig> TSessionsFetcherConfig::Registrator(TSessionsFetcherConfig::GetType());

bool TSessionsFetcher::DoFetch(const NAlerts::TFetcherContext& context) {
    CHECK_WITH_LOG(context.GetServer());
    CHECK_WITH_LOG(context.GetServer()->GetDriveAPI());
    auto sBuilder = context.GetServer()->GetDriveAPI()->GetTagsManager().GetDeviceTags().GetHistoryManager().GetSessionsBuilder("billing", context.GetFetchInstant());
    auto sessions = sBuilder->GetSessionsActual();

    if (sessions.empty()) {
        return false;
    }
    auto carsData = context.GetServer()->GetDriveAPI()->GetCarsData()->FetchInfo(TInstant::Zero());
    for (auto& session : sessions) {
        auto lastEvent = session->GetLastEvent();
        if (!lastEvent) {
            continue;
        }

        if (!Config.GetStatesFilter().empty() && !Config.GetStatesFilter().contains((*lastEvent)->GetName())) {
            continue;
        }

        if (Config.HasTagDuration() && lastEvent->GetHistoryTimestamp() > Now() - Config.GetTagDurationRef()) {
            continue;
        }

        TBillingSession::TBillingCompilation compilation;
        if (!session->FillCompilation(compilation)) {
            continue;
        }

        if (!Config.GetOffersFilter().empty() && compilation.GetCurrentOffer() && !Config.GetOffersFilter().contains(compilation.GetCurrentOffer()->GetBehaviourConstructorId()) && !Config.GetOffersFilter().contains(compilation.GetCurrentOffer()->GetPriceConstructorId())) {
            continue;
        }

        if (!Config.GetOfferTagsFilter().IsEmpty()) {
            if (!compilation.GetCurrentOffer()) {
                continue;
            }
            auto action = context.GetServer()->GetDriveAPI()->GetRolesManager()->GetAction(compilation.GetCurrentOffer()->GetBehaviourConstructorId());
            auto offerBuilder = action ? action->GetAs<IOfferBuilderAction>() : nullptr;
            if (!offerBuilder) {
                continue;
            }
            if (!Config.GetOfferTagsFilter().IsMatching(offerBuilder->GetGrouppingTags())) {
                continue;
            }
        }

        if (!Config.GetModelsFilter().empty()) {
            auto resPtr = carsData.GetResultPtr(session->GetObjectId());
            if (!resPtr || !Config.GetModelsFilter().contains(resPtr->GetModel())) {
                continue;
            }
        }
        if ((!GetAllowedUserIds().Defined() || GetAllowedUserIds()->contains(session->GetUserId()))
            && (!GetAllowedCarIds().Defined() || GetAllowedCarIds()->contains(session->GetObjectId()))
            && (!GetAllowedSessionIds().Defined() || GetAllowedSessionIds()->contains(session->GetSessionId()))) {
            Sessions.push_back(session);
        }
    }

    if (Config.GetEntityType() == NAlerts::EAlertEntityType::User) {
        auto cmp = [](const TSessionPtr& l, const TSessionPtr& r) {
            return l->GetUserId() < r->GetUserId();
        };

        std::sort(Sessions.begin(), Sessions.end(), cmp);
    } else if (Config.GetEntityType() == NAlerts::EAlertEntityType::Car) {
        auto cmp = [](const TSessionPtr& l, const TSessionPtr& r) {
            return l->GetObjectId() < r->GetObjectId();
        };
        std::sort(Sessions.begin(), Sessions.end(), cmp);
    } else {
        auto cmp = [](const TSessionPtr& l, const TSessionPtr& r) {
            return l->GetSessionId() < r->GetSessionId();
        };
        std::sort(Sessions.begin(), Sessions.end(), cmp);
    }

    return true;
}

bool TSessionsFetcherConfig::DoDeserializeFromJson(const NJson::TJsonValue& json) {
    return OfferTagsFilter.DeserializeFromString(json["offer_tags_filter"].GetString())
        && NJson::ParseField(json, "offers_filter", OffersFilter)
        && NJson::ParseField(json, "states_filter", StatesFilter)
        && NJson::ParseField(json, "models_filter", ModelsFilter)
        && NJson::ParseField(json, "tag_duration", TagDuration)
        && NJson::ParseField(json["entity_type"], NJson::Stringify(Entity));
}

NJson::TJsonValue TSessionsFetcherConfig::DoSerializeToJson() const {
    NJson::TJsonValue result;
    TJsonProcessor::WriteContainerArray(result, "offers_filter", OffersFilter);
    TJsonProcessor::Write(result, "offer_tags_filter", OfferTagsFilter.ToString());
    TJsonProcessor::WriteContainerArray(result, "states_filter", StatesFilter);
    TJsonProcessor::WriteContainerArray(result, "models_filter", ModelsFilter);
    TJsonProcessor::Write(result, "entity_type", ToString(Entity));
    NJson::InsertNonNull(result, "tag_duration", TagDuration);
    return result;
}

NDrive::TScheme TSessionsFetcherConfig::DoGetScheme(const IServerBase& server) const {
    NDrive::TScheme scheme;

    scheme.Add<TFSVariants>("offers_filter", "Фильтр офферов").SetVariants(IOfferBuilderAction::GetNames(server.GetAsPtrSafe<NDrive::IServer>())).SetMultiSelect(true);
    scheme.Add<TFSString>("offer_tags_filter", "Фильтр тегов офферов");

    TVector<TString> tags = { "old_state_reservation", "old_state_acceptance", "old_state_riding", "old_state_parking" };
    scheme.Add<TFSVariants>("states_filter", "Фильтр состояний").SetVariants(tags).SetMultiSelect(true);
    TVector<TString> entites = { ToString(NAlerts::EAlertEntityType::User), ToString(NAlerts::EAlertEntityType::Car), ToString(NAlerts::EAlertEntityType::Session) };
    scheme.Add<TFSVariants>("entity_type", "Кого выбирать").SetVariants(entites).SetMultiSelect(false);
    TSet<TString> modelIds = MakeSet(NContainer::Keys(server.GetAsPtrSafe<NDrive::IServer>()->GetDriveAPI()->GetModelsData()->GetCached().GetResult()));
    scheme.Add<TFSVariants>("models_filter", "Фильтр моделей").SetVariants(modelIds).SetMultiSelect(true);
    scheme.Add<TFSDuration>("tag_duration", "Сколько тег должен находится в текущем состоянии");
    return scheme;
}


TString TSessionsFetcherConfig::GetSchemeDescription() const {
    return "Поездки";
}

NAlerts::EDataFetcherType TSessionsFetcherConfig::GetFetcherType() const {
    return GetType();
}

NAlerts::EAlertEntityType TSessionsFetcherConfig::GetEntityType() const {
    return GetEntity();
}

NAlerts::IServiceDataFetcher::TPtr TSessionsFetcherConfig::BuildFetcher() const {
    return new TSessionsFetcher(*this);
}
