#include "interval_checker.h"


namespace NAlerts {

    TString IDataIntervalChecker::Infinity = "inf";

    IDataIntervalChecker::TPtr IDataIntervalChecker::Construct(const TString& iteratorType) {
        TPtr result = TFactory::Construct(iteratorType);
        if (!result) {
            return new TDefaultIntervalChecker();
        }
        return result;
    }

    TString IDataIntervalChecker::GetIteratorName() const {
        return IteratorName;
    }

    NDrive::TScheme IDataIntervalChecker::GetDefaultScheme() {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("min_value", "Левая граница полуинтервала [A, B)").SetDefault("0");
        scheme.Add<TFSString>("max_value", "Правая граница полуинтервала [A, B)").SetDefault("2");
        scheme.Add<TFSBoolean>("skip_empty", "Игнорировать отсутствие данных (проверять попадание в полуинтервал только при наличии значений, иначе считать, что значение хорошее: false - skip, true - valid)").SetDefault(false);
        scheme.Add<TFSString>("iterator_name", "Имя итератора для использования в подстановках");
        return scheme;
    }

    bool IDataIntervalChecker::Check(const TFetchedValue& value) const {
        if (value == NAlerts::InvalidData) {
            return SkipEmpty;
        }
        return value >= GetMinValue() && value < GetMaxValue();
    }

    TFetchedValue TDefaultIntervalChecker::GetMinValue() const {
        return MinValue;
    }

    TFetchedValue TDefaultIntervalChecker::GetMaxValue() const {
        return MaxValue.Defined() ? *MaxValue : Max<TFetchedValue>();
    }

    bool TDefaultIntervalChecker::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (json["max_value"].GetString() != Infinity) {
            TFetchedValue maxValue = 0;
            if (!NJson::ParseField(json, "max_value", maxValue, false)) {
                return false;
            }
            MaxValue = maxValue;
        }
        return NJson::ParseField(json, "min_value", MinValue, false) &&
               NJson::ParseField(json, "iterator_name", IteratorName, false) &&
               NJson::ParseField(json, "skip_empty", SkipEmpty, false);
    }

    NJson::TJsonValue TDefaultIntervalChecker::SerializeToJson() const {
        NJson::TJsonValue json;
        NJson::InsertField(json, "min_value", ToString(MinValue));
        if (!MaxValue.Defined()) {
            NJson::InsertField(json, "max_value", Infinity);
        } else {
            NJson::InsertField(json, "max_value", ToString(*MaxValue));
        }
        NJson::InsertField(json, "skip_empty", SkipEmpty);
        NJson::InsertField(json, "iterator_name", IteratorName);
        return json;
    }

    NDrive::TScheme TDefaultIntervalChecker::GetScheme() const {
        return GetDefaultScheme();
    }

    TFetchedValue TDurationIntervalChecker::GetMinValue() const {
        return MinValue.Seconds();
    }

    TFetchedValue TDurationIntervalChecker::GetMaxValue() const {
        return MaxValue.Defined() ? MaxValue->Seconds() : Max<TFetchedValue>();
    }

    bool TDurationIntervalChecker::DeserializeFromJson(const NJson::TJsonValue& json) {
        if (json["max_value"].GetString() != Infinity) {
            TDuration maxValue;
            if (!TJsonProcessor::Read(json, "max_value", maxValue, false)) {
                return false;
            }
            MaxValue = maxValue;
        }
        return TJsonProcessor::Read(json, "min_value", MinValue, false) &&
               NJson::ParseField(json, "iterator_name", IteratorName, false) &&
               NJson::ParseField(json, "skip_empty", SkipEmpty, false);
    }

    NJson::TJsonValue TDurationIntervalChecker::SerializeToJson() const {
        NJson::TJsonValue json;
        TJsonProcessor::WriteDurationString(json, "min_value", MinValue);
        if (!MaxValue.Defined()) {
            NJson::InsertField(json, "max_value", Infinity);
        } else {
            TJsonProcessor::WriteDurationString(json, "max_value", *MaxValue);
        }
        NJson::InsertField(json, "skip_empty", SkipEmpty);
        NJson::InsertField(json, "iterator_name", IteratorName);
        return json;
    }

    NDrive::TScheme TDurationIntervalChecker::GetScheme() const {
        NDrive::TScheme scheme;
        scheme.Add<TFSDuration>("min_value", "Левая граница полуинтервала [A, B), длительность").SetDefault(TDuration::Seconds(0));
        scheme.Add<TFSDuration>("max_value", "Правая граница полуинтервала [A, B), длительность").SetDefault(TDuration::Seconds(60));
        scheme.Add<TFSBoolean>("skip_empty", "Игнорировать отсутствие данных (проверять попадание в полуинтервал только при наличии значений, иначе считать, что значение хорошее: false - skip, true - valid)").SetDefault(false);
        scheme.Add<TFSString>("iterator_name", "Имя итератора для использования в подстановках");
        return scheme;
    }

    NDrive::TScheme TMoneyIntervalChecker::GetScheme() const {
        NDrive::TScheme scheme;
        scheme.Add<TFSString>("min_value", "Левая граница полуинтервала [A, B), копейки").SetDefault("0");
        scheme.Add<TFSString>("max_value", "Правая граница полуинтервала [A, B), копейки").SetDefault("2");
        scheme.Add<TFSBoolean>("skip_empty", "Игнорировать отсутствие данных (проверять попадание в полуинтервал только при наличии значений, иначе считать, что значение хорошее: false - skip, true - valid)").SetDefault(false);
        scheme.Add<TFSString>("iterator_name", "Имя итератора для использования в подстановках");
        return scheme;
    }
}

NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator1(::ToString(NAlerts::EFetchedItems::LastActionDelta));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator2(::ToString(NAlerts::EFetchedItems::SessionStateTime));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TMoneyIntervalChecker> Registrator3(::ToString(NAlerts::EFetchedItems::SessionBill));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator4(::ToString(NAlerts::EFetchedItems::RegApprovedDelta));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator5(::ToString(NAlerts::EFetchedItems::LandingAcceptanceDelta));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TMoneyIntervalChecker> Registrator6(::ToString(NAlerts::EFetchedItems::BillingTagSum));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator7(::ToString(NAlerts::EFetchedItems::MaintenanceTime));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDefaultIntervalChecker> Registrator8(::ToString(NAlerts::EFetchedItems::MaintenanceMileageDelta));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator9(::ToString(NAlerts::EFetchedItems::LastRequestTime));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator10(::ToString(NAlerts::EFetchedItems::BindMailTime));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TMoneyIntervalChecker> Registrator11(::ToString(NAlerts::EFetchedItems::SpentSum));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator12(::ToString(NAlerts::EFetchedItems::PackOfferRemainingDuration));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator13(::ToString(NAlerts::EFetchedItems::ChatLastMessageDelta));
NAlerts::IDataIntervalChecker::TFactory::TRegistrator<NAlerts::TDurationIntervalChecker> Registrator14(::ToString(NAlerts::EFetchedItems::SLADeadline));
