#pragma once

#include <drive/backend/alerts/types/alert_types.h>
#include <drive/backend/abstract/frontend.h>
#include <drive/backend/database/drive_api.h>

namespace NAlerts {

    class IDataIntervalChecker {
    public:
        using TPtr = TAtomicSharedPtr<IDataIntervalChecker>;
        using TFactory = NObjectFactory::TObjectFactory<IDataIntervalChecker, TString>;

    public:
        virtual TFetchedValue GetMinValue() const = 0;
        virtual TFetchedValue GetMaxValue() const = 0;
        TString GetIteratorName() const;

        bool Check(const TFetchedValue& value) const;

        virtual TString Format(const TFetchedValue& value, const NDrive::IServer& server, ELocalization locale) const = 0;

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) = 0;
        virtual NJson::TJsonValue SerializeToJson() const = 0;

        virtual NDrive::TScheme GetScheme() const = 0;
        static NDrive::TScheme GetDefaultScheme();

        static TPtr Construct(const TString& iteratorType);

        virtual ~IDataIntervalChecker() {}

    protected:
        static TString Infinity;
        bool SkipEmpty = false;
        TString IteratorName;
    };

    class TDefaultIntervalChecker: public IDataIntervalChecker {
    public:
        virtual TString Format(const TFetchedValue& value, const NDrive::IServer& /*frontend*/, ELocalization /*locale = ELocalization::Rus*/) const override {
            return ::ToString(value);
        }

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme() const override;

        virtual TFetchedValue GetMinValue() const override;
        virtual TFetchedValue GetMaxValue() const override;

    private:
        TFetchedValue MinValue = Min<TFetchedValue>();
        TMaybe<TFetchedValue> MaxValue;
    };

    class TDurationIntervalChecker: public IDataIntervalChecker {
    public:
        virtual TString Format(const TFetchedValue& value, const NDrive::IServer& server, ELocalization locale = ELocalization::Rus) const override {
            return server.GetLocalization()->FormatDuration(locale, TDuration::Seconds(value));
        }

        virtual bool DeserializeFromJson(const NJson::TJsonValue& json) override;
        virtual NJson::TJsonValue SerializeToJson() const override;
        virtual NDrive::TScheme GetScheme() const override;

        virtual TFetchedValue GetMinValue() const override;
        virtual TFetchedValue GetMaxValue() const override;
    private:
        TDuration MinValue = Min<TDuration>();
        TMaybe<TDuration> MaxValue;
    };

    class TMoneyIntervalChecker: public TDefaultIntervalChecker {
    public:
        virtual TString Format(const TFetchedValue& value, const NDrive::IServer& server, ELocalization locale = ELocalization::Rus) const override {
            return server.GetLocalization()->FormatPrice(locale, value);
        }

        virtual NDrive::TScheme GetScheme() const override;
    };
}
